#!/usr/bin/env perl
# Install script for Men & Mice Central
# Copyright (c) 2003-2007 Men & Mice http://www.menandmice.com
# If you encounter any problems running this script
# please contact support@bluecatnetworks.com

use strict;
use User::pwent;
use User::grent;

# initialize variables
my $installCentral=1;
my $centralChrootDir="";
my $centralDataDir="/var/mmsuite/mmcentral";
my $centralUser="root";
my $centralGroup="root";

my $installUpdater=0;
my $installCmdline=0;

my $installBin="/usr/bin";
my $installSbin="/usr/sbin";

my $isUbuntu=0;
my $platform="";
my $startupDir="/etc/init.d";
my $runlevels="";
my $numRunLevels=6;

my $rootUser="root";
my $rootGroup="root";

my $errMissingFiles="One or more files appear to be missing from this package.\nPlease re-download it from www.menandmice.com\n";

# Examine the environment and validate package contents
my $cwd = "";
chomp($cwd = `pwd`);
(my $progName) = ($0 =~ m|/([^/]+)$|);
die "Please cd to the installers directory and run \"./$progName\"\n" unless -r "$cwd/$progName";

my $CHECKLIST;
open ($CHECKLIST, "< contents.txt") or die $errMissingFiles;
while(<$CHECKLIST>) {
	chomp;
	die $errMissingFiles unless -r $_;
}
close $CHECKLIST;
my $version = "";
chomp($version = `cat VERSION`);

# Effective uid must be root
die "You need to have root privileges to run this script.\n" unless $> == 0;

print "\nWelcome to the Men & Mice Central $version installer.\n";

my $runlevelDir = "";
# Detect platform
SWITCH: for (`uname`) {
	/Linux/		&& do { 
					$platform = "linux";
					$runlevels = ("35");
					if (-f "/etc/rc.d/init.d/functions") {
						$runlevelDir = "/etc/rc.d";
					} else {
						$runlevelDir = "/etc/init.d";
					}
					if (-f "/etc/lsb-release") {
						my $searchDistrib = "(DISTRIB_ID=)(\\S+)*";
						chomp(my $lsbRelease = `egrep "$searchDistrib" /etc/lsb-release`);
						(my $ignore, my $lsbDistrib) = ($lsbRelease =~ m|$searchDistrib|);
						if ($lsbDistrib eq "Ubuntu") {
							$isUbuntu = 1;
						}
					}
					last;
				};
	/SunOS/		&& do { 
					$platform = "solaris"; 
					$runlevelDir = "/etc";
					$runlevels = ( "3" );
					$numRunLevels = 3;
					last; 
				};
	/FreeBSD/		&& do { 
					$platform = "freebsd"; 
					$installBin="/usr/local/bin";
					$installSbin="/usr/local/sbin";
					# FreeBSD uses a different scheme for startup scripts.
					$startupDir = "/usr/local/etc/rc.d";
					$runlevelDir = $startupDir;
					$runlevels = ( );
					$numRunLevels = 0;
					last; 
				};
	die "ERROR: Men & Mice Central does not support this platform.\n";
}

die "ERROR: This installer can not install the Men & Mice Central on your platform.\nPlease download the correct installer from www.menandmice.com\n" unless -d "$cwd/$platform";

# Check library dependencies
my $missingLibs = "";
chomp($missingLibs = `ldd $cwd/$platform/* | egrep -i "not found" | sort | uniq`);
if ($missingLibs ne "") {
	my $lib = "";
	my $libs = "";
	my $ignore = ""; # Declared to get rid of warning
	foreach my $line (split("\n", $missingLibs)) {
		($lib, $ignore) = split(" ", $line);
		$libs = "$libs\t$lib\n";
	}
	die "ERROR: Men & Mice Central depends on libraries missing from your system:\n$libs\nPlease contact support\@bluecatnetworks.com\n";
}

my $libs = "";
chomp($libs = `ldd $cwd/$platform/* | egrep "\\.so" | sort | uniq`);
if ($libs eq "") {
	die "ERROR: Micetro binaries appear to be incompatible with your system.\nIf this is a 64-bit system, you will need to install 32-bit compatibility\nlibraries before installing Micetro.\n";
}

# Special case for FreeBSD.  Sometimes the /usr/local/etc/rc.d directory doesn't exist
if ($platform eq "freebsd") {
	die "Directory $startupDir doesn't exist.\nPlease create it and run this installer again\n" unless -r $startupDir;
}

for my $tempDir ( $startupDir, $runlevelDir ) {
	die "ERROR: Directory $tempDir not found.\nPlease contact support\@bluecatnetworks.com\n" unless -d $tempDir;
}

# Function prototypes
sub setDefaults();
sub isLegalUserGroup($);
sub illegalUserGroup($);
sub createOwnedDir($$$);
sub setRunlevel($$$$$);
sub setFreeBSDAutolaunch($$);
sub getUsersGroup($$);
sub removeTrailingSymbol($$);
sub removeQuotes($);
sub parentDir($);
sub askUser($$);
sub askYesNo($$);
sub killProgram($);
sub getSystemCtlPath();
sub isSystemd();
sub serviceControl($$);
sub getSystemdRootPath();
sub enableService($);

# First we need to find the name of group 0 (probably either "root" or "wheel"
$rootGroup = getUsersGroup($rootUser, $rootGroup);
$centralGroup = $rootGroup;

# Set default values for questions to user
setDefaults();

my $systemdRootPath = getSystemdRootPath();

if(isSystemd()) {
    die "Unable to determine where to install systemd service files\n. Installation aborted\n." unless -d $systemdRootPath;    
}

# Show RELEASENOTES file if requested
if (askYesNo("Would you like to view the RELEASENOTES file?", "y") != 0) {
	system("less", "RELEASENOTES");
	if ($? == -1) {
		system("more", "RELEASENOTES");
		if ($? == -1) {
			print "Could not execute either `less` or `more`.\nYou will have to view the RELEASENOTES file manually outside of this installer.\n\n";
		}
	}
}


# get input from user
$installCentral = askYesNo("Do you want to install Men & Mice Central?", $installCentral);
$installCmdline = askYesNo("Do you want to install the Men & Mice Command Line Interface?", $installCmdline);

if (! $installCentral && ! $installCmdline) {
	die "\nInstallation aborted.\n";
}

if ($installCentral) {
	$installUpdater = 1;
	if ($centralChrootDir ne "") {
		if ($centralDataDir ne "" && -d "$centralChrootDir$centralDataDir" && askYesNo("Previous installation of Men & Mice Central has been found running within\na chroot() environment.  Running Men & Mice Central within a chroot()\nenvironment is not recommended and support for such configurations may be\nremoved in a later version of the Men & Mice Central.\nWould you still like to keep running Men & Mice Central within that chroot()\nenvironment?", "n")) {
			$centralDataDir = "$centralChrootDir$centralDataDir";
		} else {
			$centralDataDir = "$centralChrootDir$centralDataDir";
			$centralChrootDir = "";
		}
	}

	if ($centralChrootDir eq "") {
		$centralDataDir = askUser("Where do you want Men & Mice Central to keep its configuration\nfiles?", "$centralDataDir");
		$centralDataDir = removeTrailingSymbol($centralDataDir, "/");
	}

	# Ask user for valid "user:group" pair
	my $userGroup = "";
	do {
		$userGroup = askUser("Enter the user and group names under which you want to run\nMen & Mice Central.", "$centralUser:$centralGroup");
	} while(illegalUserGroup($userGroup));
	($centralUser, $centralGroup) = split(":", $userGroup);

	# Ask user for installation directory for system binary
	do {
		$installSbin = askUser("Where do you want to install the Men & Mice Central binary?", $installSbin);
		if (! -d $installSbin) {
			print "Directory $installSbin does not exist\n";
		}
	} while(! -d $installSbin);
}

if ($installCmdline) {
	do {
		$installBin = askUser("Where do you want to install the Men & Mice Command Line Interface?", $installBin);
		if (! -d $installBin) {
			print "Directory $installBin does not exist\n";
		}
	} while(! -d $installBin);
}

########### Done asking user.  Time to do something ###########

print "\n";

# Stop daemon if it is running
if ($installCentral && ( -f "$startupDir/mmcentral" || -f "$systemdRootPath/mmcentral.service")) {
    serviceControl("mmcentral", "stop");
    killProgram("mmcentrald");
}
if ($installCentral && ( -f "$startupDir/mmupdater" || -f "$systemdRootPath/mmupdater.service")) {
    serviceControl("mmupdater", "stop");
    killProgram("mmupdaterd");
}

# install components

if ($installCentral) {
	my $relativeDataDir = $centralDataDir;
	$relativeDataDir =~ s|^$centralChrootDir||; # remove chroot dir from path

	print "Creating directory $centralDataDir\n";
	createOwnedDir("$centralDataDir", 0755, "$centralUser:$centralGroup");
	if (! -r "$centralDataDir/mmcentral.conf") {
		system("cp", "$cwd/mmcentral.conf", "$centralDataDir/");
	}

	createOwnedDir("$centralChrootDir/var/run", 0755, "$rootUser:$rootGroup");
	createOwnedDir("$centralChrootDir/var/run/mmcentrald", 0755, "$centralUser:$centralGroup");

	print "Installing binary \"mmcentrald\" in $installSbin\n";
	system("cp", "$cwd/$platform/mmcentrald", $installSbin);

	# Copy startup script, inserting the correct parameters to mmcentrald
    if (isSystemd()) {
        print "Creating a systemd unit file for mmcentrald\n";

        my $serviceFilePath = "$cwd/startupscripts/mmcentral.service";
        if ($isUbuntu) {
            $serviceFilePath = "$cwd/startupscripts/ubuntu/mmcentral.service";
        }

        open (TEMPOUT, "> $systemdRootPath/mmcentral.service");
        print TEMPOUT `sed -e 's|_USER_|$centralUser|' -e 's|_GROUP_|$centralGroup|' -e 's|_DATA_|$relativeDataDir|'  -e 's|_PATH_|$installSbin|' $serviceFilePath`;
        close TEMPOUT;


        # If we are upgrading from a previous init.d type install, we do a clean up:
        if ( -f "$startupDir/mmcentral") {
            system("rm", "$startupDir/mmcentral");
        }

        system("chown", "-R", "$rootUser:$rootGroup", "$systemdRootPath/mmcentral.service");
        system("chmod", "644", "$systemdRootPath/mmcentral.service");
   
       
    } else {
        # INIT 
        print "Installing startup script \"mmcentral\" in $startupDir\n";
        
        if (! $isUbuntu) {
            open (TEMPOUT, "> $startupDir/mmcentral");
            print TEMPOUT `sed -e 's|_CHROOTDIR_|$centralChrootDir|' -e 's|_PARAM1_|-u $centralUser -g $centralGroup -d $relativeDataDir|' -e 's|_PATH_|$installSbin|' \"$cwd\"/startupscripts/mmcentral`;
            close TEMPOUT;

           } else { # Ubuntu
            open (TEMPOUT, "> $startupDir/mmcentral");
            print TEMPOUT `sed -e 's|_LOCATION_|$installSbin|' \"$cwd\"/startupscripts/ubuntu/mmcentral.init`;
            close TEMPOUT;
        }       
    }

    if($isUbuntu) {
        print "Installing default configuration file in /etc/default/mmcentral\n";
        open (TEMPOUT, "> /etc/default/mmcentral");
        print TEMPOUT `sed -e 's|_CHROOTDIR_|$centralChrootDir|' -e 's|_USER_|$centralUser|' -e 's|_GROUP_|$centralGroup|' -e 's|_DATA_|$relativeDataDir|' \"$cwd\"/startupscripts/ubuntu/mmcentral.default`;
        close TEMPOUT;

        system("chown", "$rootUser:$rootGroup", "/etc/default/mmcentral");
        system("chmod", "644", "/etc/default/mmcentral");
   }

	# Setup correct access privileges
    if ( ! isSystemd()) {
	    print "Setting access privileges for $centralDataDir, $startupDir/mmcentral and $installSbin/mmcentrald\n";
        system("chown", "-R", "$rootUser:$rootGroup", "$startupDir/mmcentral");
	    system("chmod", "755", "$startupDir/mmcentral");
    } else {
	    print "Setting access privileges for $centralDataDir and $installSbin/mmcentrald\n";
    }
    
    system("chown", "-R", "$centralUser:$centralGroup", "$centralDataDir");
	system("chown", "$rootUser:$rootGroup", "$installSbin/mmcentrald");
	system("chmod", "-R", "755", "$centralDataDir");
	system("chmod", "640", "$centralDataDir/mmcentral.conf");
	system("chmod", "755", "$installSbin/mmcentrald");

    enableService("mmcentral");
    

	if(-d "$cwd/update"){
		print "Installing update binaries in $centralDataDir/update\n";
		system("mkdir", "-p", "$centralDataDir/update");
		system("cp -R $cwd/update/* $centralDataDir/update/");
		print "Setting access privileges for $centralDataDir/update\n";
		system("chown", "-R", "$centralUser:$centralGroup", "$centralDataDir/update");
	}else{
		print "Update files not found in installer package, not installing automatic updates.\n";
	}
	print ("\n");
}

if ($installUpdater) {
	print "Installing binary \"mmupdaterd\" in $installSbin\n";
	system("cp", "$cwd/$platform/mmupdaterd", $installSbin);

	createOwnedDir("/var/run", 0755, "$rootUser:$rootGroup");
	createOwnedDir("/var/run/mmupdaterd", 0755, "$rootUser:$rootGroup");

	# Copy startup script, inserting the correct parameters to mmupdaterd
    if (isSystemd()) {
        print "Creating a systemd unit file for mmupdaterd\n";
        
        my $serviceFilePath = "$cwd/startupscripts/mmupdater.service";
        if ($isUbuntu) {
            $serviceFilePath = "$cwd/startupscripts/ubuntu/mmupdater.service";
        }

        open (TEMPOUT, "> $systemdRootPath/mmupdater.service");
        print TEMPOUT `sed -e 's|_USER_|$rootUser|' -e 's|_GROUP_|$rootGroup|' -e 's|_PATH_|$installSbin|' $serviceFilePath`;
        close TEMPOUT;
        
        system("chown", "$rootUser:$rootGroup", "$systemdRootPath/mmupdater.service");
	    system("chmod", "644", "$systemdRootPath/mmupdater.service");

        # If we are upgrading from a previous init.d type install, we do a clean up:
        if ( -f "$startupDir/mmupdater") {
            system("rm", "$startupDir/mmupdater");
        }

    } else {
        # Not systemd   
        print "Installing startup script \"mmupdater\" in $startupDir\n";
        
        if (! $isUbuntu) {
            open (TEMPOUT, "> $startupDir/mmupdater");
            print TEMPOUT `sed -e 's|_CHROOTDIR_||' -e 's|_PARAM1_||' -e 's|_PATH_|$installSbin|' \"$cwd\"/startupscripts/mmupdater`;
            close TEMPOUT;
        } else { # Ubuntu
            open (TEMPOUT, "> $startupDir/mmupdater");
            print TEMPOUT `sed -e 's|_LOCATION_|$installSbin|' \"$cwd\"/startupscripts/ubuntu/mmupdater.init`;
            close TEMPOUT;
        }
    }

    if ($isUbuntu) {
        print "Installing default configuration file in /etc/default/mmupdater\n";
        open (TEMPOUT, "> /etc/default/mmupdater");
        print TEMPOUT `sed -e 's|_USER_||' -e 's|_GROUP_||' \"$cwd\"/startupscripts/ubuntu/mmupdater.default`;
        close TEMPOUT;

        system("chown", "$rootUser:$rootGroup", "/etc/default/mmupdater");
        system("chmod", "644", "/etc/default/mmupdater");
    }

	# Setup correct access privileges
    if ( ! isSystemd()) {
	    print "Setting access privileges for $startupDir/mmupdater, $installSbin/mmupdaterd\n";
        system("chown", "$rootUser:$rootGroup", "$startupDir/mmupdater");
	    system("chmod", "755", "$startupDir/mmupdater");
    } else {
	    print "Setting access privileges for $installSbin/mmupdaterd\n";
    }

	system("chown", "$rootUser:$rootGroup", "$installSbin/mmupdaterd");
	system("chmod", "755", "$installSbin/mmupdaterd");

    enableService("mmupdater");
}

if ($installCmdline) {
	print "Installing binary \"mmcmd\" in $installBin\n";
	system("cp", "$cwd/$platform/mmcmd", $installBin);

	print "Setting access privileges for $installBin/mmcmd\n";
	system("chown", "$rootUser:$rootGroup", "$installBin/mmcmd");
	system("chmod", "755", "$installBin/mmcmd");

	print "\n";
}

# Launch daemon
if ($platform eq "freebsd") {
	# FreeBSD prints all daemon names on the same line, so we have to add some sort of prefix.
	$| = 1;	# Turn on auto-flushing, else this string is printed after the output from the startup scripts
	print "Launching Men & Mice Central daemon:";
}
if ($installCentral) {
    serviceControl("mmcentral", "start");
    serviceControl("mmupdater", "start");
}

print "\nInstallation of Men & Mice Central was successful.\n";

########### DONE ###########

sub setDefaults() {
	my $searchParams = "^\\w*(PARAMS=)";
	my $paramsResult = "";
	my $searchDataDir = "^\\w*(PARAMS=)[^#]+\\s+-d\\s*([^#\\s]+).*\$";
	my $dataDirResult = "";
	my $searchRootDir = "^\\w*(PARAMS=)[^#]+\\s+-r\\s*([^#\\s]+).*\$";
	my $rootDirResult = "";
	my $searchChrootDir = "^\\w*(CHROOTDIR=)(\\S+)*";
	my $chrootDirResult = "";
	my $searchConfigFile = "^\\w*(PARAMS=)[^#]+\\s+-c\\s*([^#\\s]+).*\$";
	my $configFileResult = "";
	my $ignore = ""; # Declared to get rid of warning

	if (-f "$startupDir/mmcentral") {
		my $centralScript = "$startupDir/mmcentral";

		$installCentral = 1;

		die "\nERROR: You do not have read access to $centralScript\nPlease log in as root to run this installer\n" unless -r "$centralScript";

		$paramsResult = `egrep "$searchChrootDir" $centralScript 2> /dev/null`;
		($ignore, $chrootDirResult) = ($paramsResult =~ m|$searchChrootDir|);
		if ($chrootDirResult ne "") {
			$centralChrootDir = removeQuotes($chrootDirResult);
		}

		$paramsResult = `egrep "$searchParams" $centralScript 2> /dev/null`;
		($ignore, $dataDirResult) = ($paramsResult =~ m|$searchDataDir|);
		if ($dataDirResult ne "") {
			$centralDataDir = removeQuotes($dataDirResult);
		}
	}

	if (-f "$installBin/mmcmd") {
		$installCmdline = 1;
	}
}

sub isLegalUserGroup($) {
	my @ug = split(":", $_[0]);
	if ($#ug + 1 == 2) {
		if ($ug[0] ne "" && $ug[1] ne "" && getpwnam($ug[0]) && getgrnam($ug[1])) {
			return 1;
		}
	}
	return 0;
}

sub illegalUserGroup($) {
	if (!isLegalUserGroup($_[0])) {
		print "Illegal user or group name\n";
		return 1;
	}
	return 0;
}

sub createOwnedDir($$$) {
	my ($dir, $perm, $owner) = @_;
	my $x = "";

	if (!($dir =~ m/^\//)) {
		$x = ".";
	}

	for my $d (split("/", $dir)) {
		if ($d ne "") {
			$x = "$x/$d";
			if (! -d $x) {
				mkdir $x, $perm or die "Unable to create directory \"$x\"\n";
				system("chown", "$owner", "$x");
			}
		}
	}
}

sub setRunlevel($$$$$) {
	my ($script, $scriptDir, $rcDir, $levels, $priority) = @_;

	print "Setting $scriptDir/$script to start at runlevel(s) ", join(", ", split("", $levels)), " in $rcDir\n";

	for my $x (1..$numRunLevels) {
		my $xDir = "$rcDir/rc$x.d";
		die "ERROR: Directory $xDir not found.\nPlease contact support\@bluecatnetworks.com\n" unless -d $xDir;
		# Clean up any old links
		system ("rm -f $xDir/S*$script");
		system ("rm -f $xDir/K*$script");
		# Install either start- or kill-script
		if ($levels =~ m/$x/) {
			system ("ln -s $scriptDir/$script $xDir/S$priority$script > /dev/null 2> /dev/null");
		} else {
			system ("ln -s $scriptDir/$script $xDir/K$priority$script > /dev/null 2> /dev/null");
		}
	}
}

sub setFreeBSDAutolaunch($$) {
	my ($script, $scriptDir) = @_;
	# FreeBSD only launches scripts in "/usr/local/etc/rc.d" if their names end with ".sh",
	# so all we have to do is to add a symlink with name "$script.sh".  To prevent service from
	# starting automatically, just remove the link
	
	print "Adding symbolic link $script.sh to $script in $scriptDir\n";
	die "$scriptDir/$script not found\nPlease contact support\@bluecatnetworks.com\n" unless -f "$scriptDir/$script";
	system ("ln -s $scriptDir/$script $scriptDir/$script.sh > /dev/null 2> /dev/null");
	system ("chown $rootUser:$rootGroup $scriptDir/$script $scriptDir/$script.sh");
	system ("chmod 555 $scriptDir/$script $scriptDir/$script.sh");
}

sub getUsersGroup($$) {
	my ($userName, $defGroup) = @_;

	if (my $pwEnt = getpwnam($userName)) {
		my $groupEnt;
		if ($groupEnt = getgrgid($pwEnt->gid)) {
			return $groupEnt->name;
		}
	}
	return $defGroup;
}

sub removeQuotes($) {
	my ($tmp) = (@_);
	$tmp =~ s|^"||;
	$tmp =~ s|"$||;
	return $tmp;
}

sub removeTrailingSymbol($$) {
	my ($tmp, $sym) = (@_);
	my $dir;

	do {
		$dir = $tmp;
		if ($dir eq "/") {
			return $dir;
		}

		$tmp =~ s|$sym$||;
	} while ($tmp ne $dir);

	return $dir;
}

# parentDir("/foo/bar") == "/foo";
sub parentDir($) {
	my ($dir) = (@_);
	$dir =~ s|/[^/]+/?$||;
	if ($dir eq "") {
		$dir = "/";
	}
	return $dir;
}

sub askUser($$) {
	my ($prompt, $default) = @_;
	print "\n$prompt [$default]:";
	chomp(my $reply = <STDIN>);
	if ($reply eq "") {
		$reply = $default;
	} elsif ($reply =~ m/^[qQ]$/) {
		die "\nInstallation aborted.\n";
	}
	return $reply;	
}
 
sub askYesNo($$) {
	my ($prompt, $default) = @_;

	if ($default eq 0) {
		$default = "n";
	} elsif ($default eq 1) {
		$default = "y";
	}

	my $reply = askUser($prompt, $default);
	if ($reply =~ m/^[yY]/) {
		return 1;
	}
	return 0;
}

sub killProgram($) {
	if (-f "/usr/bin/killall") {
		system("/usr/bin/killall $_[0] 2> /dev/null");
	} elsif (-f "/usr/bin/pkill") {
		system("/usr/bin/pkill -x $_[0] 2> /dev/null");
	}
}

sub isSystemd() {
    my $pgrep = "";
    if ( -f "/usr/bin/pgrep") {
        $pgrep = "/usr/bin/pgrep";             
    } else {
        die "Unable to determine if startup system type is systemd or not.\n";    
    }
    # systemd will run with a process id of 1.
    system("$pgrep systemd | grep '^1\$' > /dev/null");
    return ($? eq 0);
}

sub getSystemCtlPath() {

    if( -f "/usr/bin/systemctl") { # Red Hat
        return "/usr/bin/systemctl";    
   } elsif ( -f "/bin/systemctl") { # Ubuntu
        return "/bin/systemctl";
   } else {
        return "systemctl";       
  }
}

sub getSystemdRootPath() {
    if ( -d "/usr/lib/systemd/system") {
        return "/usr/lib/systemd/system";
    } else {
        return "/etc/systemd/system";    
    } 
}

sub serviceControl($$) {
    my ($binary, $action) = @_;
    if(isSystemd()) {
        my $systemctl = getSystemCtlPath();
        if ( -f $systemctl) {
            system($systemctl, $action, $binary);
        } else {
            die "unable to execute systemctl to start or stop services.";    
        }
    } else {
        # initd
	    system("$startupDir/$binary", $action);
    }
}

sub enableService($) {
    my ($serviceName) = @_;

    if ( ! isSystemd()) {
        # Set runlevels
        if ($isUbuntu) {
            system("update-rc.d $serviceName defaults 40 60");
        } else {
             if ($platform ne "freebsd") {
                setRunlevel($serviceName, $startupDir, $runlevelDir, $runlevels, 40);
            } else {
                setFreeBSDAutolaunch($serviceName, $startupDir);
            }
        }
    } else {
        system(getSystemCtlPath(), "daemon-reload");
        system(getSystemCtlPath(), "enable", $serviceName);
    }
}

