#!/usr/bin/env python
#
# File: install.py
# Desc: A update/installation script for Men & Mice appliance
#
#
import sys,os,time
import shutil
import getopt
import subprocess
import logging
import collections

def printHelp():
	print "A update/installation script for the Men & Mice Appliance."
	print "Usage: install.py [--keep-data][--keep-network][--keep-bootconfig][--keep-password]"
	print "                  [--replace-data][--replace-network][--replace-bootconfig][--replace-password]"
	print "                  [--no-reboot][--reboot]"
	print "                  [--debug][--log=<file>]"
	print ""


log = None

bKeepData= True					# if true we will use old data
bKeepNetwork= True				# if true we will use old network setup
bKeepPassword= True				# if true we will use old password file
bKeepBootConfig = True			# if true we will use old boot configure
bDoBackup = True				# if true it will do a backup of mydata before start updating
bDoReboot = True				# if true the script will reboot when done
bDetectIfRebootNeeded = True	# if true we will try to update without doing a reboot
try:

	opts, extraparams = getopt.getopt(sys.argv[1:], "h",["keep-data", "keep-network", "keep-bootconfig", "keep-password"
						, "replace-data", "replace-network", "replace-bootconfig", "replace-password", "no-reboot", "reboot"
						, "debug", "log="
						, "help"])

	tmpLog = logging.getLogger(__name__)
	hdlr = logging.StreamHandler(sys.stderr)
	logging.getLogger().setLevel(logging.WARNING)

	for o, p in opts:
		if o in ["--keep-data", "--replace-data"]:
			bKeepData= o in ["--keep-data"]
		elif o in ["--keep-network", "--replace-network"]:
			bKeepNetwork= o in ["--keep-network"]
		elif o in ["--keep-password", "--replace-password"]:
			bKeepPassword= o in ["--keep-password"]
		elif o in ["--keep-bootconfig", "--replace-bootconfig"]:
			bKeepBootConfig= o in ["--keep-bootconfig"]
		elif o in ["--reboot", "--no-reboot"]:
			bDoReboot= o in ["--reboot"]
			bDetectIfRebootNeeded= False
		elif o in ("--debug"):
			logging.getLogger().setLevel(logging.DEBUG)
		elif o in ("-l", "--log"):
			hdlr = logging.FileHandler(a)			
		elif o in ["-h", "--help"]:
			printHelp()
			exit(0)
		else:
			assert False, "unhandled option"

 	 				
	formatter = logging.Formatter('%(filename)s:%(lineno)s %(levelname)s %(message)s')
	hdlr.setFormatter(formatter)
	logging.getLogger().addHandler(hdlr)
	log = tmpLog

	if os.geteuid() != 0:
		raise Exception("You need to have root privileges to run this script.")


except Exception, err:
	print >> sys.stderr, err
	print >> sys.stderr, "For help use --help"
	exit(2)



# =============================================================================
# Utilities
#

def run_basic(p_cmd, failOnExit= True):
	strCommand = " ".join(p_cmd)
	try:
		log.debug( "Executing: [" + strCommand + "]" )
		prc = subprocess.Popen(p_cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
	except OSError as exc:
		log.error( "Error executing command: [" + strCommand + "] - "  + str(exc) )
		raise Exception("Error executing command: [" + strCommand + "] - "  + str(exc))

	stdoutdata, stderrdata = prc.communicate()

	if failOnExit and (prc.returncode < 0):
		log.error( "Error executing command: [" + strCommand + "] - "  + str(prc.returncode) )
		raise Exception("Error executing command: [" + strCommand + "] - "  + str(prc.returncode))

	return [prc.returncode, stdoutdata, stderrdata] 


def run_shell(p_cmd, failOnExit= True):
	strCommand = " ".join(p_cmd)
	err= subprocess.call(strCommand, shell=True)
	log.debug( "Shell run: [" + strCommand + "] - " + str(err) )
	if err != 0:
		log.error( "Error executing shell command: [" + strCommand + "] - "  + str(err) )
		raise Exception("Error executing shell command: [" + strCommand + "] - "  + str(err))


# Changes owner and access mode of the file 'inFileName'
# to 'inOwner' and 'inAccessMod' respectively
def chownmod(inOwner, inAccessMod, inFileName):
	theCommand = "chown " + inOwner + " " + inFileName + " && chmod " + inAccessMod + " " + inFileName
	run_shell( [theCommand] )

#
# Tries to open the file 'inFileName' and returns the first line in the file.
# Returns an empty string if it able to read it
def getFilesFirstLine(inFileName):
	strResult=""
	try:                                                
		with open(inFileName, "r") as f:
			strResult= f.readline().rstrip()
		f.close()       
	except Exception, err:          
		pass
	return strResult

# reads a password like file containing items on the form
# item 1:item 2:....
# to an ordered dictionary
def readPasswdFile(inFileName):
	result = collections.OrderedDict()
	with open(inFileName, 'r') as f:
		for strLine in f.readlines():
			strTmp = strLine.rstrip()
			theSplit = strTmp.split(':')
			if (len(theSplit) > 1) and len(theSplit[0]) > 0:
				result[theSplit[0]] = theSplit;
	return result

# writes a password like file form an dictionary read with
# readPasswdFile to a file
#
def writePasswdFile(inFileName, inItems):
	try:
		with open(inFileName, "w") as f:
			for key, val in inItems.iteritems():
				f.write(':'.join(val) + "\n")
	except Exception, err:
		raise Exception("error: while writing out \"" + inFileName + "\"")
	
#
# Reads the action file 'inFileName' that contains versions and actions that
# should be performed between versions. The format of the file is a follows:
#	<version>
#	{
#		<needed steps>
#		<new version> 
#	}+
#
# returns whether 'inCurrentVersion' was found, last version mentioned and the
# actions needed to move from current to new
#
def getActionItems(inFileName, inCurrentVersion):
	try:
		if len(inCurrentVersion) == 0:
			return [False, "", dict()]
			
		bFoundVersion= False
		strLatestVersion= ""
		lineNo= 1
		actions= dict()    # A dictionary of files and operations todo when doing incremental update
		with open(inFileName, "r") as f:
			for strLine in f.readlines():
				currLine= strLine.strip()
				currLineLen= len(currLine)
				if (currLineLen > 0):
					if currLine[0] == '[':
						strLatestVersion=""
						if (currLineLen > 2) and (currLine[currLineLen - 1] == ']'):
							strLatestVersion=currLine[1:currLineLen - 1].strip()
						
						if len(strLatestVersion) == 0:
							raise Exception("error: incorrect format of " + inFileName + ":" + str(lineNo) + ". Missing version number.")
				
						if (bFoundVersion == False) and (strCurrentVersion == strLatestVersion):
							bFoundVersion= True				
					elif (currLine[0] == '+') or (currLine[0] == '-'):
						if bFoundVersion:
							# yes we are collection operations
							actions[currLine[1:].strip()] = currLine[0]
					else:
						raise Exception("error: incorrect format of " + inFileName + ":" + str(lineNo) + ". Unknown entry type.")

				lineNo= lineNo + 1
			f.close()

		return [bFoundVersion, strLatestVersion, actions]

	except Exception, err:
		log.error( "Reading action file " + inFileName + " - " + str(err) )
		exit(2)


# =============================================================================
# MAIN PROGRAM: 
#
log.debug( "Updating appliance" )
 

try:
	# if we are going to keep data we should backup everything
	if bKeepData or bKeepNetwork or bKeepPassword:
		log.debug( "Backing up mydata.tgz" )
		# Ignore exit code, might return file not found 
		# and we don't care
		run_basic( ["su", "tc", "-c", "\"filetool.sh -b\""], False )
except Exception, err:
	log.error( "When doing backup - " + str(err) )


# new stuff should be in the same directory as the install program
installScriptDir= os.path.abspath(os.path.dirname(sys.argv[0]))		# contains the directory of install.py
newStuffDir= os.path.join(installScriptDir,"tc")

tceNotChanged = False


# -----------------------------------------------------------------------------
# Need to find the boot directory and the data directory.
# Boot directory should be kept in /boot/.mm_data if not check
# /opt/.tce_dir
bootPartition= getFilesFirstLine("/opt/.mm_boot")
if len(bootPartition) == 0:
	try:
		bootPartition= getFilesFirstLine("/opt/.tce_dir")
		if len(bootPartition) == 0:
			raise Exception("error: Unable to find boot directory.")
		bootPartition= os.path.abspath(bootPartition + "/..")
	except Exception, err:
		log.error( "When detecting tce directory - " + str(err) )
		exit(2)                                      

# check if we have a separate data partition.
dataPartition= getFilesFirstLine("/opt/.mm_data")
if len(dataPartition) == 0:
	dataPartition= bootPartition

dirTceCurr=bootPartition + "/tce"
dirTceBak=bootPartition + "/tce.bak"
log.debug( "TCE directory: " + dirTceCurr )


strCurrentVersion= getFilesFirstLine(os.path.join(dirTceCurr, "VERSION"))
strCurrentRevision= getFilesFirstLine(os.path.join(dirTceCurr, "REVISION"))
strNewVersion= getFilesFirstLine(os.path.join(installScriptDir,"VERSION"))
strNewRevision= getFilesFirstLine(os.path.join(installScriptDir,"REVISION"))
# 'strCurrentVersion' contains the number of the version we are currently running
# 'strCurrentRevision' contains the revision number of that we are currently running
# 'strNewVersion' contains the number of the version we want to install/upgrade to
# 'strNewRevision' contains the revision number that we want to install/upgrade to
log.debug( "Current Version: " + strCurrentVersion + "[" + strCurrentRevision + "]")
log.debug( "New Version: " + strNewVersion + "[" + strNewRevision + "]")

# -----------------------------------------------------------------------------
# Check for Incremental backup
#
# An incremental update is different from a regular update in the way that it 
# only contains the files needed to upgrade from x to y.
# If there exists a file called install.inc we will try to do an incremental update
#
installIncFileName= os.path.join(installScriptDir, "install.inc")
bDoingIncUpdate= os.path.isfile(installIncFileName)

incActions= dict()    # A dictionary of files and operations todo when doing incremental update


if bDoingIncUpdate:
	# There exist a incremental update file, let's inspect it
	log.debug( "Found incremental update file \"" + installIncFileName + "\"." )
	bFoundVersion, strLatestVersion, incActions = getActionItems(installIncFileName, strCurrentVersion)

	if (bFoundVersion == False) or (strNewVersion != strLatestVersion):
		# ok we found an incremental update file but it is not applicable
		# If the version file is there in the install we assume that this a
		# full update so we will revert to that instead of bailing out
		if len(strNewVersion) > 0:
			# The version file is there, this is a full version, do a full update
			log.debug( "Unable to do incremental update. Doing full install." )
			bDoingIncUpdate= False
		else:
			log.error( "Unable to update. This incremental update is not applicable for current version." )
			exit(3)
	
	# Now if we have a revision number we will only do a incremental update if the new revision number is
	# larger or the same
	if bDoingIncUpdate and len(strCurrentRevision) > 0 and ( len(strNewRevision) == 0 or int(strCurrentRevision) > int(strNewRevision) ):
		log.debug( "Unable to do incremental update. Current revision is newer. Doing full install." )
		bDoingIncUpdate= False
	
	if 	bDoingIncUpdate:
		log.debug( "Doing incremental upgrade " + strCurrentVersion + "[" + strCurrentRevision + "] -> " +  strNewVersion + "[" + strNewRevision + "]" )
	
# -----------------------------------------------------------------------------
# Boot partition
#
# If there exists a new boot partion we will check if we need to update it. 
# If we need to update it we will have to update no matter what.
#
# Take a backup of the current system files and copy the new files the system 
# files are: microcore64.gz and vmlinuz64 
#
# NOTE: Currently we will ignore boot directory if we are doing an incremental upgrade
#       We should however check if the are the same and if not complain about it
if (bDoingIncUpdate == False) and os.path.exists(newStuffDir + "/boot"):

	log.debug( "Updating <" + bootPartition + "/boot" + ">" )

	try:
		os.rename(bootPartition + "/boot/microcore64.gz", bootPartition + "/boot/microcore64.gz.bak")
		os.rename(bootPartition + "/boot/vmlinuz64",      bootPartition + "/boot/vmlinuz64.bak")
		shutil.copy(newStuffDir + "/boot/microcore64.gz", bootPartition + "/boot/microcore64.gz")
		chownmod("root:root", "644", bootPartition + "/boot/microcore64.gz")
		shutil.copy(newStuffDir + "/boot/vmlinuz64", bootPartition + "/boot/vmlinuz64")
		chownmod("root:root", "644", bootPartition + "/boot/vmlinuz64")

		if bKeepBootConfig != True:
			log.debug( "Updating <" + bootPartition + "/boot/extlinux/extlinux.conf" + ">" )
			os.rename(bootPartition + "/boot/extlinux/extlinux.conf", bootPartition + "/boot/extlinux/extlinux.conf.bak")
			shutil.copy(newStuffDir + "/boot/extlinux/extlinux.conf", bootPartition + "/boot/extlinux/extlinux.conf")
			chownmod("root:root", "644", bootPartition + "/boot/extlinux/extlinux.conf")

	except Exception, err:
		log.error( "Updating boot partition - " + str(err) )
		exit(2)

# if there is a difference in the new and the old files we need to restart
try:
	pass
	# tceNotChanged = filecmp.cmp(bootPartition + "/boot/microcore64.gz", bootPartition + "/boot/microcore64.gz.bak") and filecmp.cmp(bootPartition + "/boot/vmlinuz64", bootPartition + "/boot/vmlinuz64.bak")

except Exception, err:
	log.error( "Checking for changes - " + str(err) )
	# exit(2)


# -----------------------------------------------------------------------------
# TCE
#
# Update TCE directory. 
# If this is a regular update we will take a backup of the old tce directory
# and simply replace it with a new version.
# If this is a incremental update we will take a copy of the tce directory
# and apply all changes
#
log.debug( "Updating <" + dirTceCurr + "> backing up at <" + dirTceBak + ">" )


try:
	if bootPartition != dataPartition:
		# now this must be old partition stuff and in that case we need to
		# get rid of old packets directory so we will have enough space
		shutil.rmtree(bootPartition+"/mm/var/db/packets.orginal", True)

	shutil.rmtree(dirTceBak, True)

except Exception, err:
	log.error( "Removing old packets - " + str(err) )
	exit(2)


bRevertTceBak=False
try:
	if bDoingIncUpdate:

		if len(incActions) > 0:

			shutil.copytree(dirTceCurr, dirTceBak)
			bRevertTceBak=True

			for key, val in incActions.iteritems():
				if val == '+':
					log.debug( "ADD: " + os.path.join(newStuffDir, key) + " -> " + os.path.join(bootPartition, key) )
					shutil.copy( os.path.join(newStuffDir, key), os.path.join(bootPartition, key) )
				elif val == '-':
					try: 
						# if the file doesn't exist we will not complain
						log.debug( "DEL: " + os.path.join(bootPartition, key) )
						os.remove( os.path.join(bootPartition, key) )
					except Exception, err:
						# NOTE: this is just a warning we don't care if the file is already gone
						log.warning( "Unable to delete file - " + str(err) )
	
		# patch the version file to reflect the new version
		with open(dirTceCurr + "/VERSION", "w") as f:
			f.write(strNewVersion+"\n")

		# patch the revision file to reflect the new revision
		with open(dirTceCurr + "/REVISION", "w") as f:
			f.write(strNewRevision+"\n")

		if len(incActions) == 0:
			# nothing todo. We have already update the tc/tce/VERSION so we can 
			# safely exit now
			exit(0)

	else:

		shutil.move(dirTceCurr, dirTceBak)
		bRevertTceBak=True

		shutil.copytree(newStuffDir + "/tce", dirTceCurr)

except Exception, err:
	# we had an exception while trying to copy, need to revert to the old tce 
	log.error( "Updating TCE - " + str(err) )
	print >> sys.stderr, err
	if bRevertTceBak:
		log.debug( "Reverting " + dirTceBak + " -> " + dirTceCurr )
		if os.path.exists(dirTceBak):
			shutil.rmtree(dirTceCurr, True)
			shutil.move(dirTceBak, dirTceCurr)
		else:
			log.error( "When reverting to backup. Missing tce backup directory: " + dirTceBak )
		
	exit(2)



try:
	# ------------------------------------------------
	# Make sure privileges are correct
	#
	# tce should be owned by tc:staff
	# most files should be -rw-rw-r--
	# directories drwxrwxr-x
	theCommand = "chown -R tc:staff " + bootPartition + "/tce && chmod -R 664 " + bootPartition + "/tce && chmod 775 " + bootPartition + "/tce " + bootPartition + "/tce/optional/"
	run_shell( [theCommand] )

	
	# ------------------------------------------------
	# Fix mydata.tgz if needed
	# Also backup the orginal mydata.tgz so we can use
	# it at a reference later on
	#
	oldDataFile= os.path.join(bootPartition,"tce.bak/mydata.tgz")
	newDataFile= os.path.join(bootPartition,"tce/mydata.tgz")
	shutil.copy(newDataFile, newDataFile + ".bak")

	if not bDoingIncUpdate and os.path.isfile(oldDataFile) and (bKeepData or bKeepNetwork or bKeepPassword):
		#
		# We will use old mydata.tgz as basis but overwrite everything with the new mydata.tgz 
		# If we are doing incremental update we are not updating mydata.tgz
		log.debug( "Merging together <" + oldDataFile + "> and <" + newDataFile + ">" )
		tmpDir = "/tmp/newdata"
		shutil.rmtree(tmpDir, True)
		tmpNewDataDir= os.path.join(tmpDir, "mydata")
		os.makedirs(tmpNewDataDir)
		# Now we: 
		#   1. extract the old data 
		#   2. extract the new data tar
		#   3. tar the directory  
		theCommand = "cd " + tmpNewDataDir + " && tar -xzf " + oldDataFile

		# if we only wanted to keep data not network we should not extract network
		if not bKeepNetwork:
			theCommand += " --exclude opt/network" 
		elif not bKeepData:
			# only keep opt/network and in the case we want to preserve password
			# extract etc/shadow
			theCommand += " opt/network" 
			if bKeepPassword:
				theCommand += " etc/shadow" 

		if bKeepPassword:
			theCommand +=  " && mv etc/shadow etc/shadow.org "

		theCommand +=  " && tar -xzf " + newDataFile

		run_shell( [theCommand] )

		if bKeepPassword:
			# if 'bKeepPassword' is true we should have two files:
			#	etc/shadow.org 	- is the old file
			#	etc/shadow 		- is the new shadow file
			#
			# We need to create a new shadow file containing the orginal except:
			#	- remove all entries that doesn't exist in etc/passwd
			#	- add all new entries from shadow.new
			dictPasswd = readPasswdFile( 		os.path.join(tmpNewDataDir, 'etc/passwd') 		)
			dictShadowCurr = readPasswdFile( 	os.path.join(tmpNewDataDir, 'etc/shadow.org')	)
			dictShadowNew = readPasswdFile( 	os.path.join(tmpNewDataDir, 'etc/shadow')		)

			# go through all items in current shadow file and make sure they do exist in passwd file
			for key, val in dictShadowCurr.iteritems():
				if not key in dictPasswd:
					del dictShadowCurr[key]

			# go trough all the items in shadow new and if they do not exist in current add them
			for key, val in dictShadowNew.iteritems():
				if not key in dictShadowCurr:
					dictShadowCurr[key]= val

			writePasswdFile( os.path.join(tmpNewDataDir, 'etc/shadow'), dictShadowCurr)


		# now pack the merged mydata
		theCommand = "cd " + tmpNewDataDir + " && tar -zcf " + newDataFile + " *"
		run_shell( [theCommand] )

		shutil.rmtree(tmpDir, True)

	# ------------------------------------------------
	# Remove unwanted data
	#
	if not bKeepData:		
		# then we also have to clean any /mm stuff that might be kept on the data partition
		log.debug( "Removing <" + dataPartition + "/mm/" + ">" )
		shutil.rmtree(dataPartition + "/mm/", True)



except Exception, err:
	print >> sys.stderr, err
	exit(2)

log.debug( "Updating successful" )


# -----------------------------------------------------------------------------
# Check if we need to restart the appliance because of the changes
# 
kKnownModules = {
	# Module name				supervisor name		startup script
	  'bind.tcz' 			: [ 'named'				, '']
	, 'unbound.tcz' 		: [ 'unbound'			, '']
	, 'isc-dhcp.tcz' 		: [ 'dhcpd'				, '']
	, 'mmdhcpremoted.tcz' 	: [ 'mmdhcpremoted'		, '']
	, 'mmremoted.tcz' 		: [ 'mmremoted'			, '']
	, 'apache2.tcz' 		: [ ''					, '/usr/local/etc/init.d/mmweb']
	, 'mmweb.tcz' 			: [ 'mmxmlintd'			, '']
	, 'mmupdaterd.tcz' 		: [ 'mmupdaterd'		, '']
}

def moduleIsRunning(inStrSupervisor, inStrStartupScript):
	retcode = 1
	strResult = "" 

	if len(inStrSupervisor) > 0:
		retcode, strResult, _ = run_basic( ["supervisorctl", "status", inStrSupervisor] )
	elif len(inStrStartupScript) > 0:
		retcode, strResult, _ = run_basic( [inStrStartupScript, "status"] )
	
	return ("RUNNING" in strResult)

def moduleSetRunning(inStrSupervisor, inStrStartupScript, inSetRunning):
	strCommand = "start" if inSetRunning else "stop"

	if len(inStrSupervisor) > 0:
		run_shell( ["supervisorctl", strCommand, inStrSupervisor] )
	elif len(inStrStartupScript) > 0:
		run_shell( [inStrStartupScript, strCommand] )

 
def doTCEUpgrade(inTCEName, inRestartIfNeeded):
	log.debug( "doTCEUpgrade(" + inTCEName + ", " + str(inRestartIfNeeded) + ")")

	try:
		_, strModuleName = os.path.split(inTCEName)

		module = kKnownModules.get(strModuleName, None)
		if module is None:
			log.error( "Module " + strModuleName + " not found .")
			return False


		# ------------------------------------------------------
		# Check if module is running and if so stop it
		# sudo supervisorctl stop named
		#
		strSupervisor = module[0]
		strStartupScript = module[1]
		bIsModuleRunning = (len(strSupervisor) > 0) or (len(strStartupScript) > 0)
		if bIsModuleRunning:
			bIsModuleRunning= moduleIsRunning(strSupervisor, strStartupScript)
			if (bIsModuleRunning == False) and (strSupervisor == 'mmremoted'):
				# special case for mmremoted we also need to check mmremoted2
				strSupervisor= 'mmremoted2'
				bIsModuleRunning= moduleIsRunning(strSupervisor, strStartupScript)
			
			if bIsModuleRunning:
				moduleSetRunning(strSupervisor, strStartupScript, False)


		# -------------------------------------------------	-----
		# unload and load the module
		strModuleNameWithoutExt, _ = os.path.splitext(strModuleName)
		
		# When unmounting a module we might get in to problems because someone might be using it
		# Let's retry 3 time with a little sleep in between hoping that the program running has quit
		log.debug( "unloading: " + strModuleNameWithoutExt)
		bUnloadSuccess= False
		retry= 0
		while not bUnloadSuccess and (retry < 3):
			try:
				if strModuleNameWithoutExt == "bind":
					# Special case. For bind we need to kill all nslookup which might also be in use
					run_shell( ["sudo", "killall", "nslookup"] )

				run_shell( [ os.path.join(installScriptDir, "bin/tce-unload.sh") , strModuleNameWithoutExt ] )
				bUnloadSuccess= True
			except:
				log.debug( "error while unloading: " + strModuleNameWithoutExt + ". Will retry")
				time.sleep(5)
			retry= retry + 1 
		if not bUnloadSuccess:
			log.error( "Unable to unload: " + strModuleNameWithoutExt )
			return False
			                				
		run_shell( ["sudo", "su", "tc", "-c",  "\"/usr/bin/tce-load -i " + os.path.join(bootPartition, inTCEName) +"\""] )

		# ------------------------------------------------------
		# If module was running start it again
		# sudo supervisorctl start named
		#
		if bIsModuleRunning:
			moduleSetRunning(strSupervisor, strStartupScript, True)
			if not moduleIsRunning(strSupervisor, strStartupScript):
				log.error( strModuleName + " not running after restart")
	
		return True

	except Exception, err:
		# we had an exception while trying upgrade the module this is not working 
		log.error( "While upgrading \"" + inTCEName + "\" - " + str(err) )
		return False


if bDetectIfRebootNeeded:
	try:
		updateIncFileName= os.path.join(installScriptDir, "update.inc")
		if os.path.isfile(updateIncFileName):
			bFoundVersion, strLatestVersion, updateActions = getActionItems(updateIncFileName, strCurrentVersion)
			if (bFoundVersion == True) and (strNewVersion == strLatestVersion):
				# yes we found a description of how to update without rebooting
				bUpgradeSuccess = True
				for key, val in updateActions.iteritems():
					bRestartIfNeeded = val != '-'
					if not doTCEUpgrade(key, bRestartIfNeeded):
						log.error( "Failed upgrading module \"" + key + "\". Need to do a reboot to complete upgrade." )
						bUpgradeSuccess = False
						break
				if bUpgradeSuccess == True:
					# Restore mydata.tgz else it might get overwritten later on
					# and recreate banner
					log.debug( "Restoring mydata.tgz" )
					run_basic( ["su", "tc", "-c", "\"filetool.sh -r\""], False )
					
					run_basic( [ os.path.join(installScriptDir, "bin/refreshLCD.sh") ] )

					# Success upgrading all modules, we do not need a reboot
					log.debug( "Succesfully upgraded all modules. Reboot not needed." )
					bDoReboot= False
			
	except:
		pass

if bDoReboot:
	log.debug( "Doing reboot" )
	run_shell( ["reboot"] )	

