#!/bin/bash
#
# run.sh
# NOTE: assumes commands dirname and egrep are present
#
# Usage:
#  ./run.sh <service-name>
#

# Bail out if we're being passed an incorrect number of arguments
if [ $# != 1 ] ; then
	exit 1
fi

echo -----  >> /tmp/mmUpdater.log
echo `date` >> /tmp/mmUpdater.log
echo Starting update for $1 >> /tmp/mmUpdater.log

daemon=/etc/init.d/$1

isSystemd=0

if [ ! -f $daemon ] ; then
    systemctl cat $1 > /dev/null 2>&1

    if [ $? -eq 0 ] ; then
        echo systemd installation detected >> /tmp/mmUpdater.log
        isSystemd=1
    else
	    echo error: unable to find $daemon or systemd configuration files >> /tmp/mmUpdater.log
	    exit 1
    fi
fi

# Move into folder containing this script
cd `dirname "$0"`

name=
dest=
if [ $isSystemd == 0 ] ; then
    # INIT script
    # Determine binary name
    name=`egrep "^(PROGRAM=)" $daemon`
    if [ -z $name ] ; then
        # Ubuntu has NAME instead of PROGRAM
        name=`egrep "^(NAME=)" $daemon`
        if [ -z $name ] ; then
            exit 1
        fi

        name=${name:5} # Strip "NAME=" from the name
    else
        name=${name:8} # Strip "PROGRAM=" from the name
    fi

    # Determine binary location
    dest=`egrep "^(LOCATION=)" $daemon`
    if [ -z $dest ] ; then
        exit 1
    fi

    dest=${dest:9} # Strip "LOCATION=" from the destination

    if [ ${dest:0:1} == "\"" -a ${dest:(-1)} == "\"" ] ; then
            dest=${dest:1:${#dest}-2} # Strip quotes from the destination
    fi
else
    # Systemd
    name=$1d
    dest=/usr/sbin
fi

# Replace binary
echo copying the new binary >> /tmp/mmUpdater.log
mv $dest/$name $dest/$name.bak  >> /tmp/mmUpdater.log
cp $1d $dest/$name  >> /tmp/mmUpdater.log

if [ $1 == "mmdhcpremote" ]
then
    if [ -d /usr/local/bluecat ] && [ -d ./bdds ] && [ -f ./bdds/restartdhcp.sh ] 
    then
        # BDDS's restartdhcp.sh is stored under a bdds folder
        echo Copying bdds/restartdhcp.sh to /var/mmsuite/dhcp_server_controller/scripts/ >> /tmp/mmUpdater.log
        cp ./bdds/restartdhcp.sh /var/mmsuite/dhcp_server_controller/scripts/restartdhcp.sh  >> /tmp/mmUpdater.log

        # Test if DHCP service should be enabled
        jq -e '.dhcp == "1"' /var/mmsuite/updater/services.conf >> /dev/null
        retCode=$?
        if [ $retCode -eq 0 ]; then
            echo Ensuring DHCP is enabled and running >> /tmp/mmUpdater.log
            /var/mmsuite/dhcp_server_controller/scripts/restartdhcp.sh
        fi
    elif [ -f ./restartdhcp.sh ]
    then
        # regular linux system's restartdhcp.sh is stored in the root of the update package
        echo Copying restartdhcp.sh to /var/mmsuite/dhcp_server_controller/scripts/ >> /tmp/mmUpdater.log
        cp restartdhcp.sh /var/mmsuite/dhcp_server_controller/scripts/restartdhcp.sh  >> /tmp/mmUpdater.log
    fi
fi

# update the BDDS appliance scripts if we're on a BDDS
if [ $1 == "mmupdater" ] && [ -d /usr/local/bluecat ] && [ -d ./bdds/mm ]
then
    echo "Updating BDDS scripts" >> /tmp/mmUpdater.log
    cp -r ./bdds/mm/* /usr/local/mm/
fi

# If there exist a new version of rest-doc in this package then we will 
# copy it to the correct place
# NOTE: We assume that /var/mmsuite/web_services exist, which should be the case if mmws has ever been started
if [ -d ./rest-doc ] ; then
    echo "Updating REST documentation for web services " >> /tmp/mmUpdater.log
    rm -rf /var/mmsuite/web_services/doc
    cp -r ./rest-doc /var/mmsuite/web_services/doc
fi

# Update scripts in extensions folder
if [ -d ./extensions ] ; then
    echo "Updating extensions scripts " >> /tmp/mmUpdater.log
    cp -r ./extensions /var/mmsuite/mmcentral/extensions
fi

# Create scripts folder and copy README
if [ -d ./scripts ] ; then
    echo "Updating user scripts " >> /tmp/mmUpdater.log
    cp -r ./scripts /var/mmsuite/mmcentral/scripts
fi

# Start daemon
echo Restarting $1 >> /tmp/mmUpdater.log
if [ $isSystemd == 0 ] ; then
    $daemon restart  >> /tmp/mmUpdater.log
else
    systemctl restart $1 >> /tmp/mmUpdater.log
fi

# If this is the web app upgrader, run the script to upgrade it.  Note that we do this last of all because it could fail 
# on systems where only mmws is installed.  If that happens, we want the new mmws to have been started so that it will 
# report the correct version the next time it is called
if [ -d ./webapp ] ; then
    echo "Attempting web app update " >> /tmp/mmUpdater.log
    if [ -f /var/mmsuite/web_services/upgrade_web_app.sh ] ; then
        /var/mmsuite/web_services/upgrade_web_app.sh ./webapp >> /tmp/mmUpdater.log 2>&1
        echo "web app update succesful" >> /tmp/mmUpdater.log
    else
        echo "warning: Unable to find upgrade script at /var/mmsuite/web_services/upgrade_web_app.sh" >> /tmp/mmUpdater.log
    fi
fi

echo Done updating  >> /tmp/mmUpdater.log

exit 0

