#!/usr/bin/env perl
# Install script for Men & Mice Server Controllers
# Copyright (c) 2003-2015 Men & Mice http://www.menandmice.com
# If you encounter any problems running this script
# please contact support@bluecatnetworks.com

use strict;
use User::pwent;
use User::grent;

# initialize varibles
my $installDNSRemote=1;
my $dnsRemoteRootDir="";
my $dnsRemoteChrootDir="";
my $dnsRemoteUser="named";
my $dnsRemoteGroup="named";
my $namedWithinChroot=0;
my $namedDir="/var/named";
my $namedConfigFile="/etc/named.conf";
my $namedCheckConf="/usr/sbin/named-checkconf";
my $skipArrange=0;
my $needToArrange=0;
my $netConfigFile="/etc/netconfig"; # Solaris
my @alternateNamedConfigFiles=(
	  "/etc/namedb/named.conf"		# FreeBSD
	, "/etc/bind/named.conf"		# Ubuntu
);

my @alternateNamedCheckConfs=(
	  "/usr/local/sbin/named-checkconf"
);
my $forceWhenChrootAndNamedDirIsRoot = 0;

my $unboundConfigFile = "/etc/unbound/unbound.conf";
my $unboundDataDir = "/etc/unbound";
my @alternateUnboundDataDirs = (
	"/etc/unbound" # Ubuntu, Red Hat, 
	, "/var/unbound/etc" # Solaris?
);

my $dnsRemoteDataDir = "/var/mmsuite/dns_server_controller";
my $installDHCPRemote=0;

my $dhcpRemoteDataDir="/var/mmsuite/dhcp_server_controller";
my $dhcpRemoteUser="dhcpd";
my $dhcpRemoteGroup="dhcpd";
my $dhcpdConfigFile="/etc/dhcpd.conf";
my $dhcpdLeaseFile="/var/lib/dhcpd/dhcpd.leases";

my @alternateDhcpRemoteUsersAndGroups=(
	  "dhcp:dhcp"					# Ubuntu
);

my @alternateDhcpdConfigFiles=(
	  "/etc/dhcp3/dhcpd.conf"		# Ubuntu
);

my @alternateDhcpdLeaseFiles=(
	  "/var/lib/dhcp3/dhcpd.leases"		# Ubuntu
	, "/var/lib/dhcp/dhcpd.leases"		# Red Hat 4
	, "/var/lib/dhcp/db/dhcpd.leases"	# OpenSUSE 10
);

my $installUpdater=0;
my $installCmdline=1;

my $installUtil="/usr/bin";
my $installBin="/usr/bin";
my $installSbin="/usr/sbin";

my $isUbuntu=0;
my $isChrootMount=0;
my $platform="";
my $startupDir="/etc/init.d";
my $runlevels="";
my $numRunLevels=6;

my $rootUser="root";
my $rootGroup="root";

my @portsToOpen = ();

my $errMissingFiles="One or more files appear to be missing from this package.\nPlease re-download it from www.menandmice.com\n";

# Examine the enviroment and validate package contents
my $cwd = "";
chomp($cwd = `pwd`);
(my $progName) = ($0 =~ m|/([^/]+)$|);
die "Please cd to the installers directory and run \"./$progName\"\n" unless -r "$cwd/$progName";

my $CHECKLIST;
open ($CHECKLIST, "< contents.txt") or die $errMissingFiles;
while(<$CHECKLIST>) {
	chomp;
	die $errMissingFiles unless -r $_;
}
close $CHECKLIST;
my $version = "";
chomp($version = `cat VERSION`);

# Effective uid must be root
die "You need to have root privileges to run this script.\n" unless $> == 0;

# Check if the user is setting up BIND or Generic DNS remotes
my $isDNSRemoteBIND=1;
my $isDNSRemoteGeneric=0;
my $isDNSRemoteUnbound=0;

foreach my $argnum (0 .. $#ARGV) {
	if ($ARGV[$argnum] eq "--generic") {
		$isDNSRemoteGeneric=1;
		$isDNSRemoteBIND=0;
		$isDNSRemoteUnbound=0;
		last;
	}
}

if ($isDNSRemoteGeneric) {
	print "\nWelcome to the Men & Mice $version generic DNS Server Controller installer.\n";
} else {
	print "\nWelcome to the Men & Mice $version Server Controller installer.\n";
}

my $runlevelDir = "";
# Detect platform
SWITCH: for (`uname`) {
	/Linux/		&& do { 
					$platform = "linux";
					$runlevels = ("35");
					if (-f "/etc/rc.d/init.d/functions") {
						$runlevelDir = "/etc/rc.d";
					} else {
						$runlevelDir = "/etc/init.d";
					}
					if (-f "/etc/lsb-release") {
						my $searchDistrib = "(DISTRIB_ID=)(\\S+)*";
						chomp(my $lsbRelease = `egrep "$searchDistrib" /etc/lsb-release`);
						(my $ignore, my $lsbDistrib) = ($lsbRelease =~ m|$searchDistrib|);
						if ($lsbDistrib eq "Ubuntu") {
							$isUbuntu = 1;
						}
					}
					last;
				};
	/SunOS/		&& do { 
					$platform = "solaris"; 
					$runlevelDir = "/etc";
					$runlevels = ( "3" );
					$numRunLevels = 3;
					last; 
				};
	/FreeBSD/		&& do { 
					$platform = "freebsd"; 
					$installUtil="/usr/local/bin";
					$installBin="/usr/local/bin";
					$installSbin="/usr/local/sbin";
					# FreeBSD uses a different scheme for startup scripts.
					$startupDir = "/usr/local/etc/rc.d";
					$runlevelDir = $startupDir;
					$runlevels = ( );
					$numRunLevels = 0;
					last; 
				};
	die "ERROR: Men & Mice Server Controllers do not support this platform.\n";
}

die "ERROR: This installer can not install the Men & Mice Server Controllers on your platform.\nPlease download the correct installer from www.menandmice.com\n" unless -d "$cwd/$platform";

# Check library dependancies, except for mmedit and mmlock since they're deprecated
my $missingLibs = "";
chomp($missingLibs = `ldd $cwd/$platform/mmdhcpremoted $cwd/$platform/mmremoted $cwd/$platform/mmupdaterd $cwd/$platform/arrange | egrep -i "not found" | sort | uniq`);
if ($missingLibs ne "") {
	my $lib = "";
	my $libs = "";
	my $ignore = ""; # Declared to get rid of warning
	foreach my $line (split("\n", $missingLibs)) {
		($lib, $ignore) = split(" ", $line);
		$libs = "$libs\t$lib\n";
	}
	die "ERROR: Men & Mice Server Controllers depend on libraries missing from your system:\n$libs\nPlease contact support\@bluecatnetworks.com\n";
}

my $libs = "";
chomp($libs = `ldd $cwd/$platform/* | egrep "\\.so" | sort | uniq`);
if ($libs eq "") {
	die "ERROR: Micetro binaries appear to be incompatible with your system.\nIf this is a 64-bit system, you will need to install 32-bit compatibility\nlibraries before installing Micetro.\n";
}

if (-x "/usr/sbin/getenforce") {
	chomp(my $enforceLevel = `/usr/sbin/getenforce`);
	die "ERROR: SELinux must be disabled or in permissive mode in order to be able to\ninstall Men & Mice Server Controllers." unless $enforceLevel ne "Enforcing";
}

# Special case for FreeBSD.  Sometimes the /usr/local/etc/rc.d directory doesn't exist
if ($platform eq "freebsd") {
	die "Directory $startupDir doesn't exist.\nPlease create it and run this installer again\n" unless -r $startupDir;
}

for my $tempDir ( $startupDir, $runlevelDir ) {
	die "ERROR: Directory $tempDir not found.\nPlease contact support\@bluecatnetworks.com\n" unless -d $tempDir;
}

# Function prototypes
sub setDefaults();
sub parseNamedConf($$);
sub isLegalUserGroup($);
sub illegalUserGroup($);
sub dirContainsLinks($);
sub detectNamedUser($$);
sub getAlternateIfFound($@);
sub getAlternateDirIfFound($@);
sub createDir($$);
sub createOwnedDir($$$);
sub setRunlevel($$$$$);
sub setFreeBSDAutolaunch($$);
sub getUsersGroup($$);
sub getUIDAndGID($$);
sub isSystemDirectory($);
sub removeTrailingSymbol($$);
sub removeQuotes($);
sub parentDir($);
sub getChrootDir($);
sub askUser($$);
sub askYesNo($$);
sub killProgram($);
sub commaSeperatedArr($);
sub getSystemCtlPath();
sub isSystemd();
sub serviceControl($$);
sub getSystemdRootPath();
sub enableService($);
sub sprintFile(\%$$);

# Find name of user and group that named runs as
# First we need to find the name of group 0 (probably either "root" or "wheel"
$rootGroup = getUsersGroup($rootUser, $rootGroup);
($dnsRemoteUser, $dnsRemoteGroup) = detectNamedUser($platform, $startupDir);

if (!isLegalUserGroup("$dhcpRemoteUser:$dhcpRemoteGroup")) {
	($dhcpRemoteUser, $dhcpRemoteGroup) = ($rootUser, $rootGroup);
	for my $userGroup (@alternateDhcpRemoteUsersAndGroups) {
		if (isLegalUserGroup($userGroup)) {
			($dhcpRemoteUser, $dhcpRemoteGroup) = split(":", $userGroup);
			last;
		}
	}
}

# Set default values for questions to user
setDefaults();

my $systemdRootPath = getSystemdRootPath();

if(isSystemd()) {
	die "Unable to determine where to install systemd service files\n. Installation aborted\n." unless -d $systemdRootPath;    
}


# get input from user
$installDNSRemote = askYesNo("Do you want to install the Men & Mice DNS Server Controller?", $installDNSRemote);

my $selectedRemoteInstall = "";
if($installDNSRemote && ! $isDNSRemoteGeneric) {
   do {
	   $isDNSRemoteBIND = 0;
	   $isDNSRemoteUnbound = 0;
	   $selectedRemoteInstall = askUser("Press \"B\" to install a DNS server controller for BIND or \"U\" to install a DNS server controller for Unbound:", "B");
	   if ($selectedRemoteInstall =~ m/^[bB]/) {
		   $isDNSRemoteUnbound = 0;
		   print "\nDNS Server Controller for BIND selected for installation.\n";
		   $isDNSRemoteBIND=1;
	   } elsif ($selectedRemoteInstall =~ m/^[uU]/) {
		   print "\nDNS Server Controller for Unbound selected for installation.\n";
		   $isDNSRemoteUnbound=1;
	   } else {
		  print "Unknown option \"$selectedRemoteInstall\"\n";
	   }
   } while( ! $isDNSRemoteBIND && ! $isDNSRemoteUnbound);
}

$installDHCPRemote = askYesNo("Do you want to install the Men & Mice DHCP Server Controller?", $installDHCPRemote);

if ($platform eq "solaris") {
	# mmcmd is deprecated on Solaris
	$installCmdline = 0;
} else {
	$installCmdline = askYesNo("Do you want to install the Men & Mice Command Line Interface?", $installCmdline);
}

if (! $installDNSRemote && ! $installDHCPRemote && ! $installCmdline) {
	die "\nInstallation aborted.\n";
}



########### User questions for Bind DNS controller	###########

if ($installDNSRemote && $isDNSRemoteBIND) {
	if ($dnsRemoteChrootDir ne "") {
		if ($namedConfigFile ne "" && -f "$dnsRemoteChrootDir$namedConfigFile" && askYesNo("Previous installation of Men & Mice DNS Server Controller found running\nwithin a chroot() environment.  Running Men & Mice DNS Server Controller\nwithin a chroot() environment is not recommended and support for such\nconfigurations may be removed in a later version of the Men & Mice DNS Server\nController.\nRunning Men & Mice DNS Server Controller within a chroot() environment may\nalso reduce functionality.\nWould you still like to keep running Men & Mice DNS Server Controller within\nthat chroot() environment?", "n")) {
			$namedConfigFile = "$dnsRemoteChrootDir$namedConfigFile";
			$dnsRemoteRootDir = "";
		} else {
			$dnsRemoteChrootDir = "";
		}
	}

	if ($dnsRemoteChrootDir eq "") {
		$namedConfigFile = getAlternateIfFound($namedConfigFile, @alternateNamedConfigFiles);

		$namedWithinChroot = askYesNo("Are you running named in a chroot() environment?", "n");
		if ($namedWithinChroot) {
			if (-f "/etc/SuSE-release") {
				my $suseVersion = `egrep "VERSION = 1" /etc/SuSE-release 2> /dev/null`;
				chomp($suseVersion);
				if ($suseVersion ne "") {
					die "\nDISCLAIMER:\nThe Men & Mice DNS Server Controller for Linux can NOT currently be\ninstalled when named is running in a chroot() environment and the Linux\ndistribution is OpenSuSE 10.1 or later.\nTo install the software, named must first be configured with\nNAMED_RUN_CHROOTED=\"no\" in /etc/sysconfig/named, using an editor or\nthe yast tool.\nSee Men & Mice Knowledge Hub at http://www.menandmice.com or contact\nMen & Mice Support at support\@bluecatnetworks.com for further details.\n" unless $suseVersion eq "VERSION = 10.0";
				}
			}

			$dnsRemoteRootDir = getChrootDir($dnsRemoteRootDir);
		} else {
			$dnsRemoteRootDir = "";
		}

		my $namedConfigFileX = "";
		my $outsideRoot = 0;
		do {
			$namedConfigFileX = askUser("Where is the BIND configuration file?", "$dnsRemoteRootDir$namedConfigFile");
			if (! -f $namedConfigFileX) {
						print "File not found.	Are you sure that BIND is installed on this machine?\n";
			} else {
				if ($dnsRemoteRootDir ne "") {
					my $relativeConfigFile = $namedConfigFileX;
					$relativeConfigFile =~ s|$dnsRemoteRootDir||;
					$outsideRoot = ($relativeConfigFile eq $namedConfigFileX);

					if ($outsideRoot) {
						$namedWithinChroot = askYesNo("Your BIND configuration file \"$namedConfigFileX\" is not in directory \"$dnsRemoteRootDir\".\nAre you sure that named is running in that chroot() environment?", "n");
						if (! $namedWithinChroot) {
							$dnsRemoteRootDir = "";
						}
					}
				}
			}
		} while (! -f $namedConfigFileX || ($dnsRemoteRootDir ne "" && $outsideRoot));
		$namedConfigFile = $namedConfigFileX;
	}

	# The Server Controller will not run on Solaris unless "/etc/netconfig" is present
	$netConfigFile = "$dnsRemoteChrootDir$netConfigFile";
	if(! -f $netConfigFile && $platform eq "solaris"){
		if($dnsRemoteChrootDir eq ""){
					die("The DNS Server Controller will not run on Solaris without a netconfig file. Please make sure that $netConfigFile exists and try again.\n\nInstallation aborted.\n");
			}else{
					die("The DNS Server Controller will not run on Solaris without a netconfig file in the chroot environment. Please make sure that $netConfigFile exists and try again.\n\nInstallation aborted.\n");
			}
		}
	
	if ($isDNSRemoteBIND) {
		$skipArrange = askYesNo("Would you like to skip rearranging your BIND configuration files?", "n");
		if ($skipArrange) {
			$needToArrange = 0;
			$namedDir = askUser("Where is the BIND working directory (\"directory\" option) located?", $namedDir);
			$namedDir = removeTrailingSymbol($namedDir, "/");
		} else {
			if (-f "$namedConfigFile.bak" && !askYesNo("Do you want to overwrite the backup file $namedConfigFile.bak?", "y")) {
				die "\nInstallation aborted.\n";
			}
			($needToArrange, $namedDir) = parseNamedConf($namedConfigFile, $namedDir);
		}

		die "\nUnable to parse $namedConfigFile or unable to find \"directory\" directive.\n\nInstallation aborted\n" unless $namedDir ne "";

		die "\n$namedDir is a system directory and, therefore, cannot be used.\nPlease change the \"directory\" directive of $namedConfigFile and try again.\n\nInstallation aborted\n" unless !($needToArrange && !$namedWithinChroot && isSystemDirectory($namedDir));

		if (($platform eq "linux") && ($dnsRemoteRootDir ne "")) {
			chomp(my $namedDirMount = `df -a | grep "^$namedDir "`);
			if ($namedDirMount ne "") {
				$namedConfigFile =~ s|^$dnsRemoteRootDir||; # remove root dir from path
				$dnsRemoteRootDir = "";
				$isChrootMount = 1;
			}
		}

		my $originalNamedDir = $namedDir;
		$namedDir = "$dnsRemoteRootDir$dnsRemoteChrootDir$namedDir";

		die "Directory $namedDir does not exist\nInstallation aborted.\n" unless -d $namedDir;

		die "Please move the installer directory \"$cwd\" from the \"$namedDir\" directory and run the installer again.\n" unless $cwd !~ m|^$namedDir|;

		$namedCheckConf = getAlternateIfFound($namedCheckConf, @alternateNamedCheckConfs);
		if ($dnsRemoteChrootDir eq "" && askYesNo("Would you like the DNS Server Controller to run named-checkconf\nto verify changes when editing advanced server and zone options?", "y")) {
			my $checkConf = "";
			my $useCheckConf = 1;
			do {
				$checkConf = askUser("Where is the named-checkconf binary?", "$namedCheckConf");
				if (! -f $checkConf) {
					print "Binary $checkConf does not exist\n";

					$useCheckConf = askYesNo("Would you still like the DNS Server Controller to run named-checkconf\nto verify changes when editing advanced server and zone options?", "n");
				}
			} while ($useCheckConf && ! -f $checkConf);

			if ($useCheckConf) {
				$namedCheckConf = $checkConf;

				if ($dnsRemoteRootDir ne "") {
					if (askYesNo("In order to run named-checkconf inside the chroot() environment,\nnamed-checkconf must be executable in superuser mode (setuid).\nWould you like superuser execution privilege to be set for named-checkconf?", "y")) {
						system("chmod", "+s", "$namedCheckConf");
					} else {
						print "Warning: named-checkconf may not be able to enter the chroot() environment\n";
					}
				}
			} else {
				$namedCheckConf = "";
			}
		} else {
			$namedCheckConf = "";
		}

		if ($needToArrange) {
			if ($originalNamedDir eq "/") {
				die "\n$namedDir cannot be used for the \"directory\" directive unless named is running in a chroot.\nPlease change the the \"directory\" directive or run named in a chroot.\n" unless !($dnsRemoteRootDir eq "");
				$forceWhenChrootAndNamedDirIsRoot = askYesNo("Please make sure you have a backup of \"$namedDir\" before proceeding!\nThe installer cannot recover in case of an error because the \"directory\" directive is \"$originalNamedDir\".\nIs it OK to proceed?", "n");
				die "Aborting nonrecoverable action. Please backup \"$namedDir\" or change the \"directory\" directive and run the installer again." unless $forceWhenChrootAndNamedDirIsRoot;
			} else {
				if ($isChrootMount) {
					if (!askYesNo("The installer needs to stop BIND and rearrange the files in $namedDir\nA backup of $namedDir will be created and BIND will be started again\nafterwards.  Is this OK?", "y")) {
						die "\nInstallation aborted.\n";
					}
				} else {
					if (!askYesNo("The installer needs to rearrange the files in $namedDir\nand restart the name server. A backup will be created.	Is this OK?", "y")) {
						die "\nInstallation aborted.\n";
					}
				}
				if (-d "$namedDir.bak" && !askYesNo("Do you want to overwrite the backup in $namedDir.bak?", "y")) {
					die "\nInstallation aborted.\n";
				}
			}
			if (dirContainsLinks($namedDir) && !askYesNo("The directory \"$namedDir\" contains links.\nPlease see https://docs.menandmice.com/pages/viewpage.action?pageId=6361093 for more info.\nProceed at your own risk.	Do you want to continue?", "n")) {
				die "\nInstallation aborted.\n";
			}
		}

		# Ask user for valid "user:group" pair
		my $userGroup = "";
		do {
			$userGroup = askUser("Enter the user and group names under which you want to run\nthe Men & Mice DNS Server Controller.\nThis must be the user which is running named.", "$dnsRemoteUser:$dnsRemoteGroup");
		} while(illegalUserGroup($userGroup));
		($dnsRemoteUser, $dnsRemoteGroup) = split(":", $userGroup);

		# Ask user for installation directory for utilities
		do {
			$installUtil = askUser("Where would you like to install the Men & Mice external static\nzone handling utilities?", $installUtil);
			if (! -d $installUtil) {
				print "Directory $installUtil does not exist\n";
			}
		} while (! -d $installUtil);
	}
}

########### User questions for Unbound DNS controller  ###########

if ($installDNSRemote && $isDNSRemoteUnbound) {

	# Ask user for valid "user:group" pair
	
	if ( ! illegalUserGroup("unbound:unbound")) {
		$dnsRemoteUser = "unbound";
		$dnsRemoteGroup = "unbound";
	} else {
		$dnsRemoteUser = "root";
		$dnsRemoteGroup = "root";
	}
	
	my $userGroup = "";
	do {
		$userGroup = askUser("Enter the user and group names under which you want to run the Men & Mice DNS Server Controller.\nThis must be the user which is running unbound.", "$dnsRemoteUser:$dnsRemoteGroup");
	} while(illegalUserGroup($userGroup));
	($dnsRemoteUser, $dnsRemoteGroup) = split(":", $userGroup);

	$unboundDataDir = getAlternateDirIfFound($unboundDataDir, @alternateUnboundDataDirs);
	my $unboundDataDirX = "";
	do {
		$unboundDataDirX = askUser("Where is the Unbound data dir? This is the directory where the \"unbound.conf\" configuration file is stored.\nPlease note that the ownership of the directory and its entire content will changed to the user \"$dnsRemoteUser\" and group \"$dnsRemoteGroup\"", $unboundDataDir); 
		if ( ! -d $unboundDataDirX) {
			print "\n$unboundDataDirX: No such directory.\n";
		}
		
		if ( -d $unboundDataDirX && ! -f "$unboundDataDirX/unbound.conf") {
			print "\nThe Unbound configuration file \"unbond.conf\" was not found in the specified directory \"$unboundDataDirX\".\nAre you sure that an Unbound server is installed on this machine?\n";
		}
	} while ( ! -f "$unboundDataDirX/unbound.conf");
	$unboundDataDir = removeTrailingSymbol($unboundDataDirX, "/");
	$unboundConfigFile = "$unboundDataDir/unbound.conf";

	$dnsRemoteDataDir = askUser("Where do you want the Men & Mice DNS Server Controller\nto keep its configuration files?", "$dnsRemoteDataDir");
	$dnsRemoteDataDir = removeTrailingSymbol($dnsRemoteDataDir, "/");
}

########### User questions for Generic DNS controller  ###########
if ($installDNSRemote && $isDNSRemoteGeneric) {

	# Ask user for valid "user:group" pair
	my $userGroup = "";
	do {
		$userGroup = askUser("Enter the user and group names under which you want to run\nthe Men & Mice DNS Server Controller.\n", "root:root");
	} while(illegalUserGroup($userGroup));
	($dnsRemoteUser, $dnsRemoteGroup) = split(":", $userGroup);

	$dnsRemoteDataDir = askUser("Where do you want the Men & Mice DNS Server Controller\nto keep its configuration files?", "$dnsRemoteDataDir");
	$dnsRemoteDataDir = removeTrailingSymbol($dnsRemoteDataDir, "/");
}

########### User questions for DHCP Server Controller  ###########

if ($installDHCPRemote) {

	$dhcpdConfigFile = getAlternateIfFound($dhcpdConfigFile, @alternateDhcpdConfigFiles);

	my $dhcpdConfigFileX = "";
	do {
		$dhcpdConfigFileX = askUser("Where is the DHCP server configuration file?", "$dhcpdConfigFile");
		if (! -f $dhcpdConfigFileX) {
			print "File not found.	Are you sure that a DHCP server is installed on this machine?\n";
		}
	} while (! -f $dhcpdConfigFileX);
	$dhcpdConfigFile = $dhcpdConfigFileX;

	# TODO: parse file referenced by $dhcpdConfigFile for a lease-file-name property?
	$dhcpdLeaseFile = getAlternateIfFound($dhcpdLeaseFile, @alternateDhcpdLeaseFiles);

	my $dhcpdLeaseFileX = "";
	do {
		$dhcpdLeaseFileX = askUser("Where is the DHCP server lease file?", "$dhcpdLeaseFile");
		if (! -f $dhcpdLeaseFileX) {
			print "File not found.\n";
		}
	} while (! -f $dhcpdLeaseFileX);
	$dhcpdLeaseFile = $dhcpdLeaseFileX;

	$dhcpRemoteDataDir = askUser("Where do you want the Men & Mice DHCP Server Controller\nto keep its configuration files?", "$dhcpRemoteDataDir");
	$dhcpRemoteDataDir = removeTrailingSymbol($dhcpRemoteDataDir, "/");

	# We run the DHCP controller as root, so we can restart ISC
	$dhcpRemoteUser = $rootUser;
	$dhcpRemoteGroup = $rootGroup;
}

if ($installDNSRemote || $installDHCPRemote) {
	# Ask user for installation directory for system binaries
	do {
		$installSbin = askUser("Where do you want to install the Men & Mice Server Controller binaries?", $installSbin);
		if (! -d $installSbin) {
			print "Directory $installSbin does not exist\n";
		}
	} while(! -d $installSbin);
}

if ($installCmdline) {
	do {
		$installBin = askUser("Where do you want to install the Men & Mice Command Line Interface?", $installBin);
		if (! -d $installBin) {
			print "Directory $installBin does not exist\n";
		}
	} while(! -d $installBin);
}

########### Done asking user.  Time to do something ###########

print "\n";

$installUpdater = ($installDNSRemote || $installDHCPRemote);

# Stop daemons if they are running
if ($installDNSRemote && (-f "$startupDir/mmremote" || -f "$systemdRootPath/mmremote.service")) {
	serviceControl("mmremote", "stop");
	killProgram("mmremoted");
}

if ($installDHCPRemote && (-f "$startupDir/mmdhcpremote" || -f "$systemdRootPath/mmdhcpremote.service")) {
	serviceControl("mmdhcpremote", "stop");
	killProgram("mmdhcpremoted");
}

if ($installUpdater && (-f "$startupDir/mmupdater" || -f "$systemdRootPath/mmupdater.service")) {
	serviceControl("mmupdater", "stop");
	killProgram("mmupdaterd");
}

if ($needToArrange && $isChrootMount) {
	serviceControl("named", "stop");
}

# install components

if ($isDNSRemoteBIND && $needToArrange) {
	my $relativeConfigFile = $namedConfigFile;
	$relativeConfigFile =~ s|^$dnsRemoteRootDir||; # remove root dir from path

	# Create temporary directory for arrange utility
	my $tmpDirLocation = "";
	if ($dnsRemoteRootDir ne "") {
		if (! -d "$dnsRemoteRootDir/tmp") {
			system("mkdir", "$dnsRemoteRootDir/tmp");
		}
		$tmpDirLocation = "$dnsRemoteRootDir/tmp";
	} else {
		$tmpDirLocation = parentDir($namedDir);
	}

	system("rm", "-rf", "$tmpDirLocation/named.mmsuite");
	system("rm", "-rf", "$namedDir.bak");
	createDir("$tmpDirLocation/named.mmsuite/mmsuite", 0775);	

	if ($dnsRemoteRootDir ne "") {
		if ($forceWhenChrootAndNamedDirIsRoot) {
			system("$cwd/$platform/arrange", "-c", $relativeConfigFile, "-q", "/tmp/named.mmsuite", "-t", $dnsRemoteRootDir, "-f");
		} else {
			system("$cwd/$platform/arrange", "-c", $relativeConfigFile, "-q", "/tmp/named.mmsuite", "-t", $dnsRemoteRootDir);
		}
	} else {
		system("$cwd/$platform/arrange", "-c", $namedConfigFile, "-q", "$tmpDirLocation/named.mmsuite");
	}

	die "\nInstallation aborted\n" if ($? != 0);

	my ($uid, $gid) = getUIDAndGID($dnsRemoteUser, $dnsRemoteGroup);
	if (($uid != 0) || ($gid != 0)) {
		# UID and GID in control statement in conf/user_after are 0 by default.  If 
		# $dnsRemoteUser is not root, we need to replace them with the correct UID / GID.
		`sed 's/owner 0 group 0/owner $uid group $gid/g' $namedDir/conf/user_after > $namedDir/conf/user_after~ && mv $namedDir/conf/user_after~ $namedDir/conf/user_after`;
	}

	system("touch", "$namedDir/mmsuite.log");

	if ($isChrootMount && -d "$namedDir.bak/chroot") {
		system("cp", "-a", "$namedDir.bak/chroot", "$namedDir/");
	}

	print "\n";
}

if ($installDNSRemote) {
	
	### Common to all DNS remote installations ###

	createOwnedDir("$dnsRemoteChrootDir/var/run", 0755, "$rootUser:$rootGroup");
	createOwnedDir("$dnsRemoteChrootDir/var/run/mmremoted", 0755, "$dnsRemoteUser:$dnsRemoteGroup");

	print "Installing binary \"mmremoted\" in $installSbin\n";
	system("cp", "$cwd/$platform/mmremoted", $installSbin);

	# Make sure that data directory exists
	if ($isDNSRemoteBIND && !$skipArrange) {
		$dnsRemoteDataDir = "$namedDir/mmsuite";
	}
	if (! -d $dnsRemoteDataDir) {
		system("mkdir", "-p", $dnsRemoteDataDir);
	}

	###### BIND Specific stuff ######

	if($isDNSRemoteBIND) {
		my $relativeConfigFile = $namedConfigFile;
		$relativeConfigFile =~ s|^$dnsRemoteRootDir$dnsRemoteChrootDir||; # remove (ch)root dir from path
		print "Installing binaries \"mmedit\" and \"mmlock\" in $installUtil\n";
		system("cp", "$cwd/$platform/mmedit", $installUtil);
		system("cp", "$cwd/$platform/mmlock", $installUtil);

		# Copy startup script, inserting the correct parameters to mmremoted
		if (isSystemd()) {
			print "Creating a systemd unit file for mmremoted\n";

			my $serviceFilePath = "$cwd/startupscripts/mmremote.service";
			if ($isUbuntu) {
				$serviceFilePath = "$cwd/startupscripts/ubuntu/mmremote.service";
			}

			my %configurations = (
				'_USER_' =>$dnsRemoteUser,
				'_GROUP_' => $dnsRemoteGroup,
				'_CONF_' => $relativeConfigFile,
				'_PATH_' => $installSbin,
			);
			sprintFile(%configurations, $serviceFilePath, "$systemdRootPath/mmremote.service");

				
		} else {
			# INIT
			print "Installing startup script \"mmremote\" in $startupDir\n";
	 
			if ( ! $isUbuntu) {

				my %configurations = (
				'_CHROOTDIR_' => "",
				'_PATH_' => $installSbin,
				);

				if ($dnsRemoteRootDir ne "") {
					$configurations{'_PARAM1_'} = "-u $dnsRemoteUser -g $dnsRemoteGroup -r $dnsRemoteRootDir -c $relativeConfigFile";
				} else {
					$configurations{'_PARAM1_'} = "-u $dnsRemoteUser -g $dnsRemoteGroup -c $relativeConfigFile";
					$configurations{'_CHROOTDIR_'} = $dnsRemoteChrootDir;
				}
				sprintFile(%configurations, "$cwd/startupscripts/mmremote", "$startupDir/mmremote");
			} else { # Ubuntu
				my %ubuntuInitConf = ('_LOCATION_' => $installSbin,); # No chroot stuff?
				sprintFile(%ubuntuInitConf, "$cwd/startupscripts/ubuntu/mmremote.init", "$startupDir/mmremote");
			}
		}

		if ($isUbuntu) {
			print "Installing default configuration file in /etc/default/mmremote\n";

			my %ubuntuDefaults = (
				'_CHROOTDIR_' => "",
				'_USER_' => $dnsRemoteUser,
				'_GROUP_' => $dnsRemoteGroup,
				'_ROOTDIR_' => "",
				'_CONF_' => $relativeConfigFile,
				'_DATA_' => "",
			);
			if ($dnsRemoteRootDir ne "") {
				$ubuntuDefaults{'_ROOTDIR_'} = $dnsRemoteRootDir;
			} else {
				$ubuntuDefaults{'_CHROOTDIR_'} = $dnsRemoteChrootDir;
			}

			sprintFile(%ubuntuDefaults, "$cwd/startupscripts/ubuntu/mmremote.default", "/etc/default/mmremote");
		}

		# Clear named-checkconf entry from preferences file
		my $preferencesFile = "$dnsRemoteDataDir/preferences.cfg";
		if (-f $preferencesFile) {
			open (TEMPOUT, "> $preferencesFile~");
			print TEMPOUT `grep -v '<named-checkconf value' $preferencesFile`;
			close TEMPOUT;

			system("mv", "$preferencesFile~", $preferencesFile);
		}

		# Add new named-checkconf entry to preferences file
		if ($namedCheckConf ne "") {
			open (TEMPOUT, ">> $preferencesFile");
			print TEMPOUT "<named-checkconf value=\"$namedCheckConf\"/>";
			close TEMPOUT;
		}

		# Setup correct access privileges for BIND specific config
		print "Setting access privileges for $namedDir and $namedConfigFile\n";
		system("chown", "-R", "$dnsRemoteUser:$dnsRemoteGroup", $namedDir, $namedConfigFile);
		system("chmod", "-R", "o-rwx", $namedDir);
		system("chmod", "640", $namedConfigFile);

	} # END BIND Specific configurations

	###### Unbound Specific actions ######

	if ($isDNSRemoteUnbound) {
		my %configurations = (
				'_USER_' => $dnsRemoteUser,
				'_GROUP_' => $dnsRemoteGroup,
				'_CONF_' => $unboundConfigFile,
				'_DATA_' => $dnsRemoteDataDir,
				'_PATH_' => $installSbin,
				'_LOCATION_' => $installSbin,
				'_ROOTDIR_' => "",
				'_CHROOTDIR_' => "",
		);

		if(isSystemd()) {
			print "Creating a systemd unit file for mmremoted\n";

			my $serviceFilePath = "$cwd/startupscripts/mmremote.unbound.service";
			if ($isUbuntu) {
				$serviceFilePath = "$cwd/startupscripts/ubuntu/mmremote.unbound.service";
			}

			sprintFile(%configurations, $serviceFilePath, "$systemdRootPath/mmremote.service");

		} else {
			# INIT
			print "Installing startup script \"mmremote\" in $startupDir\n";
	 
			if ( ! $isUbuntu) {
				$configurations{'_PARAM1_'} = "-u $dnsRemoteUser -g $dnsRemoteGroup -d $dnsRemoteDataDir -c $unboundConfigFile";
				sprintFile(%configurations, "$cwd/startupscripts/mmremote", "$startupDir/mmremote");
			} else { # Ubuntu
				my %ubuntuInitConf = ('_LOCATION_' => $installSbin,);
				sprintFile(%ubuntuInitConf, "$cwd/startupscripts/ubuntu/mmremote.init", "$startupDir/mmremote");
			}
		}

		if($isUbuntu) {
			print "Installing default configuration file in /etc/default/mmremote\n";
			sprintFile(%configurations, "$cwd/startupscripts/ubuntu/mmremote.default", "/etc/default/mmremote");
		}

		if( -d $unboundDataDir) {
			# Make sure that we don't the ownership of the "/etc" directory, if this is a strange setup where the config is stored there (e.g. /etc/unbound.conf).
			if ($unboundDataDir eq "/etc") {
				print "\nUnable to change ownership of $unboundDataDir. Please set the ownership of $unboundDataDir manually to $dnsRemoteUser:$dnsRemoteGroup or reconfigure unbound and run the installer again.\n\n";
			} else {
				print "Setting access privileges for $unboundDataDir\n";
				system("chown", "-R", "$dnsRemoteUser:$dnsRemoteGroup", $unboundDataDir);
				system("chmod", "-R", "o-rwx", $unboundDataDir);	   
			}
			
		}

	}
	
	##### Generic DNS server specific actions ######
	
	if($isDNSRemoteGeneric) {

		my %configurations = (
			'_USER_' => $dnsRemoteUser,
			'_GROUP_' => $dnsRemoteGroup,
			'_DATA_' => $dnsRemoteDataDir,
			'_PATH_' => $installSbin,
			'_CONF_' => "",
			'_CHROOTDIR_' => "",
			'_ROOTDIR_' => "",
			'_LOCATION_' => $installSbin,
		);
   
		if(isSystemd()) {
			print "Creating a systemd unit file for mmremoted\n";

			my $serviceFilePath = "$cwd/startupscripts/mmremote.generic.service";
			if ($isUbuntu) {
				$serviceFilePath = "$cwd/startupscripts/ubuntu/mmremote.generic.service";
			}
			sprintFile(%configurations, $serviceFilePath, "$systemdRootPath/mmremote.service");

		} else {
			# INIT
			print "Installing startup script \"mmremote\" in $startupDir\n";
	 
			if ( ! $isUbuntu) {
				$configurations{'_PARAM1_'} = "-u $dnsRemoteUser -g $dnsRemoteGroup -d $dnsRemoteDataDir";
				sprintFile(%configurations, "$cwd/startupscripts/mmremote", "$startupDir/mmremote");
			} else { # Ubuntu
				my %ubuntuInitConf = ('_LOCATION_' => $installSbin,);
				sprintFile(%ubuntuInitConf, "$cwd/startupscripts/ubuntu/mmremote.init", "$startupDir/mmremote");
			}
		}

		if($isUbuntu) {
			print "Installing default configuration file in /etc/default/mmremote\n";
			sprintFile(%configurations, "$cwd/startupscripts/ubuntu/mmremote.default", "/etc/default/mmremote");
		}
	}

	##### Common to all DNS Remotes #####
	
	if( ! isSystemd()) {
		print "Setting access privileges for $startupDir/mmremote\n";
		system("chown", "-R", "$rootUser:$rootGroup", "$startupDir/mmremote");
		system("chmod", "755", "$startupDir/mmremote");
	} else {
		# INIT
		print "Setting access privilages for $systemdRootPath/mmremote.service\n";
		system("chown", "-R", "$rootUser:$rootGroup", "$systemdRootPath/mmremote.service");
		system("chmod", "644", "$systemdRootPath/mmremote.service");
	}

	if($isUbuntu) {
		print "Setting access privilages for /etc/default/mmremote\n";
		system("chown", "$rootUser:$rootGroup", "/etc/default/mmremote");
		system("chmod", "644", "/etc/default/mmremote");
	}

	print "Setting access privileges for $installSbin/mmremoted\n";
	system("chown", "$rootUser:$rootGroup", "$installSbin/mmremoted");
	system("chmod", "755", "$installSbin/mmremoted");

	if( -f $dnsRemoteDataDir) {
		print "Setting access privileges for $dnsRemoteDataDir\n";
		system("chown", "-R", "$dnsRemoteUser:$dnsRemoteGroup", $dnsRemoteDataDir);
		system("chmod", "-R", "o-rwx", $dnsRemoteDataDir);	  
	}





	if ( isSystemd() && -f "$startupDir/mmremote") {
		# Cleanup old init.d related data
		system("rm", "$startupDir/mmremote");
	}

	enableService("mmremote");

	print "\n";
	push(@portsToOpen, 1337);
}


if ($installDHCPRemote) {
	print "Installing binary \"mmdhcpremoted\" in $installSbin\n";
	system("cp", "$cwd/$platform/mmdhcpremoted", $installSbin);

	createOwnedDir("/var/run", 0755, "$rootUser:$rootGroup");
	createOwnedDir("/var/run/mmdhcpremoted", 0755, "$dhcpRemoteUser:$dhcpRemoteGroup");

	print "Creating directory $dhcpRemoteDataDir\n";
	createOwnedDir("$dhcpRemoteDataDir", 0755, "$dhcpRemoteUser:$dhcpRemoteGroup");

	system("cp", "-r" , "$cwd/scripts", $dhcpRemoteDataDir);
	system("chown", "-R", "$dhcpRemoteUser:$dhcpRemoteGroup", "$dhcpRemoteDataDir/scripts");
	system("chmod", "-R", "755", "$dhcpRemoteDataDir/scripts");

	# Copy startup script, inserting the correct parameters to mmdhcpremoted
	if (isSystemd()) {
		print "Creating a systemd unit file for mmdhcpremoted\n";

		my $serviceFilePath = "$cwd/startupscripts/mmdhcpremote.service";
		if ($isUbuntu) {
			$serviceFilePath = "$cwd/startupscripts/ubuntu/mmdhcpremote.service";
		}

		open (TEMPOUT, "> $systemdRootPath/mmdhcpremote.service");
		print TEMPOUT `sed -e 's|_USER_|$dhcpRemoteUser|' -e 's|_GROUP_|$dhcpRemoteGroup|' -e 's|_CONF_|$dhcpdConfigFile|' -e 's|_LEASE_|$dhcpdLeaseFile|' -e 's|_DATA_|$dhcpRemoteDataDir|'  -e 's|_PATH_|$installSbin|' $serviceFilePath`;
		close TEMPOUT;

		# If we are upgrading from a previous init.d type install, we do a clean up:
		if ( -f "$startupDir/mmdhcpremote") {
			system("rm", "$startupDir/mmdhcpremote");
		}

		system("chown", "-R", "$rootUser:$rootGroup", "$systemdRootPath/mmdhcpremote.service");
		system("chmod", "644", "$systemdRootPath/mmdhcpremote.service");

	} else {
		# INIT
		print "Installing startup script \"mmdhcpremote\" in $startupDir\n";
		
		if (! $isUbuntu) {
			open (TEMPOUT, "> $startupDir/mmdhcpremote");
			print TEMPOUT `sed -e 's|_CHROOTDIR_||' -e 's|_PARAM1_|-u $dhcpRemoteUser -g $dhcpRemoteGroup -c $dhcpdConfigFile -e $dhcpdLeaseFile -d $dhcpRemoteDataDir|' -e 's|_PATH_|$installSbin|' \"$cwd\"/startupscripts/mmdhcpremote`;
			close TEMPOUT;

		} else { # Ubuntu
			if (! -f $dhcpdLeaseFile) {
				$dhcpdLeaseFile = "";
			}

			open (TEMPOUT, "> $startupDir/mmdhcpremote");
			print TEMPOUT `sed -e 's|_LOCATION_|$installSbin|' \"$cwd\"/startupscripts/ubuntu/mmdhcpremote.init`;
			close TEMPOUT;
		}
	}
	
	if ($isUbuntu) {
		print "Installing default configuration file in /etc/default/mmdhcpremote\n";
		open (TEMPOUT, "> /etc/default/mmdhcpremote");
		print TEMPOUT `sed -e 's|_USER_|$dhcpRemoteUser|' -e 's|_GROUP_|$dhcpRemoteGroup|' -e 's|_CONF_|$dhcpdConfigFile|' -e 's|_LEASE_|$dhcpdLeaseFile|' -e 's|_DATA_|$dhcpRemoteDataDir|' \"$cwd\"/startupscripts/ubuntu/mmdhcpremote.default`;
		close TEMPOUT;

		system("chown", "$rootUser:$rootGroup", "/etc/default/mmdhcpremote");
		system("chmod", "644", "/etc/default/mmdhcpremote");
	}

	# Setup correct access privileges
	if ( ! isSystemd()) {	
		print "Setting access privileges for $dhcpRemoteDataDir, $startupDir/mmdhcpremote, $dhcpdConfigFile and $installSbin/mmdhcpremoted\n";
		system("chown", "-R", "$rootUser:$rootGroup", "$startupDir/mmdhcpremote");
		system("chmod", "755", "$startupDir/mmdhcpremote");
	} else {
		print "Setting access privileges for $dhcpRemoteDataDir, $dhcpdConfigFile and $installSbin/mmdhcpremoted\n";
	}
	system("chown", "-R", "$dhcpRemoteUser:$dhcpRemoteGroup", $dhcpRemoteDataDir, $dhcpdConfigFile);
	system("chown", "$rootUser:$rootGroup", "$installSbin/mmdhcpremoted");
	system("chmod", "-R", "755", "$dhcpRemoteDataDir");
	system("chmod", "755", "$installSbin/mmdhcpremoted");
	system("chmod", "640", $dhcpdConfigFile);
	
	enableService("mmdhcpremote");

	print "\n";
	push(@portsToOpen, 4151);
}

if ($installUpdater) {
	print "Installing binary \"mmupdaterd\" in $installSbin\n";
	system("cp", "$cwd/$platform/mmupdaterd", $installSbin);

	createOwnedDir("/var/run", 0755, "$rootUser:$rootGroup");
	createOwnedDir("/var/run/mmupdaterd", 0755, "$rootUser:$rootGroup");

	# Copy startup script, inserting the correct parameters to mmupdaterd
	if (isSystemd()) {
		print "Creating a systemd unit file for mmupdaterd\n";
		
		my $serviceFilePath = "$cwd/startupscripts/mmupdater.service";
		if ($isUbuntu) {
			$serviceFilePath = "$cwd/startupscripts/ubuntu/mmupdater.service";
		}

		open (TEMPOUT, "> $systemdRootPath/mmupdater.service");
		print TEMPOUT `sed -e 's|_USER_|$rootUser|' -e 's|_GROUP_|$rootGroup|' -e 's|_PATH_|$installSbin|'	$serviceFilePath`;
		close TEMPOUT;
		
		system("chown", "$rootUser:$rootGroup", "$systemdRootPath/mmupdater.service");
		system("chmod", "644", "$systemdRootPath/mmupdater.service");


		# If we are upgrading from a previous init.d type install, we do a clean up:
		if ( -f "$startupDir/mmupdater") {
			system("rm", "$startupDir/mmupdater");
		}
	
	} else {
		if (! $isUbuntu) {
			print "Installing startup script \"mmupdater\" in $startupDir\n";
			open (TEMPOUT, "> $startupDir/mmupdater");
			print TEMPOUT `sed -e 's|_CHROOTDIR_||' -e 's|_PARAM1_||' -e 's|_PATH_|$installSbin|' \"$cwd\"/startupscripts/mmupdater`;
			close TEMPOUT;

		} else { # Ubuntu
			print "Installing startup script \"mmupdater\" in $startupDir\n";
			open (TEMPOUT, "> $startupDir/mmupdater");
			print TEMPOUT `sed -e 's|_LOCATION_|$installSbin|' \"$cwd\"/startupscripts/ubuntu/mmupdater.init`;
			close TEMPOUT;
		}
	}	 
	
	if ($isUbuntu) {
		print "Installing default configuration file in /etc/default/mmupdater\n";

		open (TEMPOUT, "> /etc/default/mmupdater");
		print TEMPOUT `sed -e 's|_USER_||' -e 's|_GROUP_||' \"$cwd\"/startupscripts/ubuntu/mmupdater.default`;
		close TEMPOUT;

		system("chown", "$rootUser:$rootGroup", "/etc/default/mmupdater");
		system("chmod", "644", "/etc/default/mmupdater");
	}
		
	# Setup correct access privileges
	if( ! isSystemd()) {
		print "Setting access privileges for $startupDir/mmupdater, $installSbin/mmupdaterd\n";
		system("chown", "$rootUser:$rootGroup", "$startupDir/mmupdater");
		system("chmod", "755", "$startupDir/mmupdater");
	} else {
		print "Setting access privileges for $installSbin/mmupdaterd\n";
	}
	system("chown", "$rootUser:$rootGroup", "$installSbin/mmupdaterd");
	system("chmod", "755", "$installSbin/mmupdaterd");

	enableService("mmupdater");

	push(@portsToOpen, 4603);
}

if ($isDNSRemoteBIND && $installCmdline) {
	print "Installing binary \"mmcmd\" in $installBin\n";
	system("cp", "$cwd/$platform/mmcmd", $installBin);

	print "Setting access privileges for $installBin/mmcmd\n";
	system("chown", "$rootUser:$rootGroup", "$installBin/mmcmd");
	system("chmod", "755", "$installBin/mmcmd");

	print "\n";
}

# Launch daemons
if ($platform eq "freebsd") {
	# FreeBSD prints all daemon names on the same line, so we have to add some sort of prefix.
	$| = 1;	# Turn on auto-flushing, else this string is printed after the output from the startup scripts
	print "Launching Men & Mice Server Controller daemons:";
}

if ($installDNSRemote && ! $isDNSRemoteGeneric) {
	serviceControl("mmremote", "start");
}
if ($installDHCPRemote) {
	serviceControl("mmdhcpremote", "start");
}
if ($installUpdater) {
	serviceControl("mmupdater", "start");
}

print "\nInstallation of the Men & Mice Server Controllers was successful.\n";

# We only need to restart named if we moved any of its files
if ($needToArrange) {
	if ($isChrootMount) {
		system("$startupDir/named", "start");
	} elsif (($platform eq "linux") && askYesNo("BIND needs to be restarted.  Would you like to restart it now?", 1)) {
		system("$startupDir/named", "restart");
	} else {
		print "Please restart BIND manually to complete the installation.\n";
	}
}

if ($installDNSRemote && $isDNSRemoteBIND && $namedCheckConf ne "" && $dnsRemoteUser ne $rootUser && $dnsRemoteGroup ne $rootGroup) {
	print "\nNOTE: $namedCheckConf needs to be executable by $dnsRemoteUser:$dnsRemoteGroup\n\n";
}

if ($#portsToOpen != -1) {
	print "Make sure that ports " . commaSeperatedArr(\@portsToOpen) . " on this machine\nare accessible from the machine running Men & Mice Central\n\n";
}
########### DONE ###########

sub setDefaults() {
	my $searchParams = "^\\w*(PARAMS=)";
	my $paramsResult = "";
	my $searchDataDir = "^\\w*(PARAMS=)[^#]+\\s+-d\\s*([^#\\s]+).*\$";
	my $dataDirResult = "";
	my $searchRootDir = "^\\w*(PARAMS=)[^#]+\\s+-r\\s*([^#\\s]+).*\$";
	my $rootDirResult = "";
	my $searchChrootDir = "^\\w*(CHROOTDIR=)(\\S+)*";
	my $chrootDirResult = "";
	my $searchConfigFile = "^\\w*(PARAMS=)[^#]+\\s+-c\\s*([^#\\s]+).*\$";
	my $configFileResult = "";
	my $ignore = ""; # Declared to get rid of warning

	if (-f "$startupDir/mmremote") {
		my $remoteScript = "$startupDir/mmremote";

		$installDNSRemote = 1;

		die "\nERROR: You do not have read access to $remoteScript\nPlease log in as root to run this installer\n" unless -r "$remoteScript";

		$paramsResult = `egrep "$searchChrootDir" $remoteScript 2> /dev/null`;
		($ignore, $chrootDirResult) = ($paramsResult =~ m|$searchChrootDir|);
		if ($chrootDirResult ne "") {
			$dnsRemoteChrootDir = removeQuotes($chrootDirResult);
			$dnsRemoteRootDir = $dnsRemoteChrootDir;
		}

		$paramsResult = `egrep "$searchParams" $remoteScript 2> /dev/null`;
		($ignore, $rootDirResult) = ($paramsResult =~ m|$searchRootDir|);
		if ($rootDirResult ne "") {
			$dnsRemoteRootDir = removeQuotes($rootDirResult);
		}

		($ignore, $configFileResult) = ($paramsResult =~ m|$searchConfigFile|);
		if ($configFileResult ne "") {
			$namedConfigFile = removeQuotes($configFileResult);
		}
	}

	if (-f "$startupDir/mmdhcpremote") {
		my $remoteScript = "$startupDir/mmdhcpremote";

		$installDHCPRemote = 1;

		die "\nERROR: You do not have read access to $remoteScript\nPlease log in as root to run this installer\n" unless -r "$remoteScript";

		$paramsResult = `egrep "$searchParams" $remoteScript 2> /dev/null`;
		($ignore, $dataDirResult) = ($paramsResult =~ m|$searchDataDir|);
		if ($dataDirResult ne "") {
			$dhcpRemoteDataDir = removeQuotes($dataDirResult);
		}

		($ignore, $configFileResult) = ($paramsResult =~ m|$searchConfigFile|);
		if ($configFileResult ne "") {
			$dhcpdConfigFile = removeQuotes($configFileResult);
		}
	}

	if (-f "$startupDir/mmupdater") {
		$installUpdater = 1;
	}

	if (-f "$installBin/mmcmd") {
		$installCmdline = 1;
	}
}

sub parseNamedConf($$) {
	my ($file, $namedDir) = @_;
	my $isMMSuite = 0;
	die "\nERROR: You do not have read/write access to $file\nPlease log in as root to run this installer\n" unless -r $file && -w $file;
	my $CONFFILE;
	open ($CONFFILE, "< $file") or die "ERROR: unable to open $file\n";
	while (<$CONFFILE>) {
		chomp;
		if (m|^// WARNING: \*\* DO NOT EDIT \*\*|) {
			$isMMSuite = 1;
		}
	}
	close $CONFFILE;
	if ($dnsRemoteRootDir ne "" || $dnsRemoteChrootDir ne "") {
		$file =~ s|^$dnsRemoteRootDir$dnsRemoteChrootDir||;	# Snip (ch)root dir from path of $file
		chomp($namedDir = qx("$cwd/$platform/arrange" -c "$file" -t "$dnsRemoteRootDir$dnsRemoteChrootDir" -p));
	} else {
		chomp($namedDir = qx("$cwd/$platform/arrange" -c "$file" -p));
	}
	
	$namedDir = removeTrailingSymbol($namedDir, "/");

	return (! $isMMSuite, $namedDir);
}

sub isLegalUserGroup($) {
	my @ug = split(":", $_[0]);
	if ($#ug + 1 == 2) {
		if ($ug[0] ne "" && $ug[1] ne "" && getpwnam($ug[0]) && getgrnam($ug[1])) {
			return 1;
		}
	}
	return 0;
}

sub illegalUserGroup($) {
	if (!isLegalUserGroup($_[0])) {
		print "Illegal user or group name\n";
		return 1;
	}
	return 0;
}

sub dirContainsLinks($) {
	my $dir= $_[0];
	chomp(my $x = `find $dir -type l`);
	if ($x eq "") {
		return 0;
	}
	return 1;
}

sub detectNamedUser($$) {
	my ($platform, $startupDir) = @_;

	my $namedLine = "";
	my $searchStr = "";

	if ($platform ne "freebsd") {
		# Grep "-u X" param from named startup script.	If param is found, find users group through system calls
		my $namedScript = "";
		if ($platform eq "solaris") {
			$namedScript = "/etc/init.d/inetsvc";
		} else {
			if ($isUbuntu) {
				$namedScript = "/etc/default/bind9";
			} else {
				$namedScript = "$startupDir/named";
			}
		}
		$searchStr = "^[^#]*(named|NAMED_BIN|PROG|OPTIONS)[^#]+-u[ \t]*(\\w+)";
		if (-r $namedScript) {
			$namedLine=`egrep "$searchStr" $namedScript 2> /dev/null`;
		}
	} else {
		# In FreeBSD, we are looking for a 'named_flags="-u bind -g bind"' line in /etc/rc.conf or /etc/defaults/rc.conf
		$searchStr = "^[^#]*(named_flags)[^#]+-u[ \t]*(\\w+)";
		for my $fileName ("/etc/rc.conf", "/etc/defaults/rc.conf" ) {
			if ($namedLine eq "" && -r $fileName) {
				$namedLine = `egrep "$searchStr" $fileName 2> /dev/null`;
			}
		}
	}
	if ($namedLine ne "") {
		my $ignore = ""; # Must declare to get rid of warning
		my $user = "";
		if (($ignore, $user) = ($namedLine =~ m|$searchStr|)) {
			my $group = getUsersGroup($user, $rootGroup);
			return ($user, $group);
		}
	}
	return ($rootUser, $rootGroup);
}

sub getAlternateIfFound($@) {
	my ($default, @alternatives) = @_;

	if (! -f $default) {
		for my $x (@alternatives) {
			if (-f $x) {
				return $x;
			}
		}
	}
	return $default;
}

sub getAlternateDirIfFound($@) {
	my ($default, @alternatives) = @_;

	if (! -d $default) {
		for my $x (@alternatives) {
			if (-d $x) {
				return $x;
			}
		}
	}
	return $default;
}

sub createDir($$) {
	my ($dir, $perm) = @_;
	my $x = "";

	if (!($dir =~ m/^\//)) {
		$x = ".";
	}

	for my $d (split("/", $dir)) {
		if ($d ne "") {
			$x = "$x/$d";
			if (! -d $x) {
				mkdir $x, $perm or die "Unable to create directory \"$x\"\n";
			}
		}
	}
}

sub createOwnedDir($$$) {
	my ($dir, $perm, $owner) = @_;
	my $x = "";

	if (!($dir =~ m/^\//)) {
		$x = ".";
	}

	for my $d (split("/", $dir)) {
		if ($d ne "") {
			$x = "$x/$d";
			if (! -d $x) {
				mkdir $x, $perm or die "Unable to create directory \"$x\"\n";
				system("chown", "$owner", "$x");
			}
		}
	}
}

sub setRunlevel($$$$$) {
	my ($script, $scriptDir, $rcDir, $levels, $priority) = @_;

	print "Setting $scriptDir/$script to start at runlevel(s) ", join(", ", split("", $levels)), " in $rcDir\n";

	for my $x (1..$numRunLevels) {
		my $xDir = "$rcDir/rc$x.d";
		die "ERROR: Directory $xDir not found.\nPlease contact support\@bluecatnetworks.com\n" unless -d $xDir;
		# Clean up any old links
		system ("rm -f $xDir/S*$script");
		system ("rm -f $xDir/K*$script");
		# Install either start- or kill-script
		if ($levels =~ m/$x/) {
			system ("ln -s $scriptDir/$script $xDir/S$priority$script > /dev/null 2> /dev/null");
		} else {
			system ("ln -s $scriptDir/$script $xDir/K$priority$script > /dev/null 2> /dev/null");
		}
	}
}

sub setFreeBSDAutolaunch($$) {
	my ($script, $scriptDir) = @_;
	# FreeBSD only launches scripts in "/usr/local/etc/rc.d" if their names end with ".sh",
	# so all we have to do is to add a symlink with name "$script.sh".	To prevent service from
	# starting automatically, just remove the link
	
	print "Adding symbolic link $script.sh to $script in $scriptDir\n";
	die "$scriptDir/$script not found\nPlease contact support\@bluecatnetworks.com\n" unless -f "$scriptDir/$script";
	system ("ln -s $scriptDir/$script $scriptDir/$script.sh > /dev/null 2> /dev/null");
	system ("chown $rootUser:$rootGroup $scriptDir/$script $scriptDir/$script.sh");
	system ("chmod 555 $scriptDir/$script $scriptDir/$script.sh");
}

sub getUsersGroup($$) {
	my ($userName, $defGroup) = @_;

	if (my $pwEnt = getpwnam($userName)) {
		my $groupEnt;
		if ($groupEnt = getgrgid($pwEnt->gid)) {
			return $groupEnt->name;
		}
	}
	return $defGroup;
}

sub getUIDAndGID($$) {
	my ($userName, $groupName) = @_;

	my $userID = 0;
	my $groupID = 0;

	if (my $pwEnt = getpwnam($userName)) {
		$userID = $pwEnt->uid;
	}

	if (my $grEnt = getgrnam($groupName)) {
		$groupID = $grEnt->gid;
	}

	return ($userID, $groupID);
}

sub isSystemDirectory($) {
	my $dir = removeTrailingSymbol($_[0], "/");

	for my $systemDirectory ("/", "/etc", "/var") {
		if ($dir eq $systemDirectory) {
			return 1;
		}
	}

	return 0;
}

sub removeQuotes($) {
	my ($tmp) = (@_);
	$tmp =~ s|^"||;
	$tmp =~ s|"$||;
	return $tmp;
}

sub removeTrailingSymbol($$) {
	my ($tmp, $sym) = (@_);
	my $dir;

	do {
		$dir = $tmp;
		if ($dir eq "/") {
			return $dir;
		}

		$tmp =~ s|$sym$||;
	} while ($tmp ne $dir);

	return $dir;
}

# parentDir("/foo/bar") == "/foo";
sub parentDir($) {
	my ($dir) = (@_);
	$dir =~ s|/[^/]+/?$||;
	if ($dir eq "") {
		$dir = "/";
	}
	return $dir;
}

sub getChrootDir($) {
	my $defaultChrootDir = $_[0];
	my $chrootDir = "";

	do {
		$chrootDir = askUser("What is the chroot() directory?", $defaultChrootDir);
		if (! -d $chrootDir) {
			print "Directory \"$chrootDir\" not found.\n";
		}
	} while (! -d $chrootDir);

	$chrootDir = removeTrailingSymbol($chrootDir, "/");
	return $chrootDir;
}

sub askUser($$) {
	my ($prompt, $default) = @_;
	print "\n$prompt [$default]:";
	chomp(my $reply = <STDIN>);
	if ($reply eq "") {
		$reply = $default;
	} elsif ($reply =~ m/^[qQ]$/) {
		die "\nInstallation aborted.\n";
	}
	return $reply;	
}
 
sub askYesNo($$) {
	my ($prompt, $default) = @_;

	if ($default eq 0) {
		$default = "n";
	} elsif ($default eq 1) {
		$default = "y";
	}

	my $reply = askUser($prompt, $default);
	if ($reply =~ m/^[yY]/) {
		return 1;
	}
	return 0;
}

sub killProgram($) {
	if (-f "/usr/bin/killall") {
		system("/usr/bin/killall $_[0] 2> /dev/null");
	} elsif (-f "/usr/bin/pkill") {
		system("/usr/bin/pkill -x $_[0] 2> /dev/null");
	}
}

# commaSeperatedArr((1,2,3,4)) == "1, 2, 3 and 4"
sub commaSeperatedArr($) {
	my @arr = @{$_[0]};
	my $str = join(", ", @arr);
	$str =~ s/, ([^,]*)$/ and \1/;
	return $str;
}


sub isSystemd() {
	my $pgrep = "";
	if ( -f "/usr/bin/pgrep") {
		$pgrep = "/usr/bin/pgrep";			   
	} else {
		die "Unable to determine if startup system type is systemd or not.\n";	  
	}
	# systemd will run with a process id of 1.
	system("$pgrep systemd | grep '^1\$' > /dev/null");
	return ($? eq 0);
}

sub getSystemCtlPath() {

	if( -f "/usr/bin/systemctl") { # Red Hat
		return "/usr/bin/systemctl";	
   } elsif ( -f "/bin/systemctl") { # Ubuntu
		return "/bin/systemctl";
   } else {
		return "systemctl";		  
  }
}

sub getSystemdRootPath() {
	if ( -d "/usr/lib/systemd/system") {
		return "/usr/lib/systemd/system";
	} else {
		return "/etc/systemd/system";	 
	} 
}


sub serviceControl($$) {
	my ($binary, $action) = @_;
	if(isSystemd()) {
		my $systemctl = getSystemCtlPath();
		if ( -f $systemctl) {
			system($systemctl, $action, $binary);
		} else {
			die "unable to execute systemctl to start or stop services.";	 
		}
	} else {
		# initd
		system("$startupDir/$binary", $action);
	}
}
sub enableService($) {
	my ($serviceName) = @_;

	if ( ! isSystemd()) {
		# Set runlevels
		if ($isUbuntu) {
			system("update-rc.d $serviceName defaults 40 60");
		} else {
			 if ($platform ne "freebsd") {
				setRunlevel($serviceName, $startupDir, $runlevelDir, $runlevels, 40);
			} else {
				setFreeBSDAutolaunch($serviceName, $startupDir);
			}
		}
	} else {
		system(getSystemCtlPath(), "daemon-reload");
		system(getSystemCtlPath(), "enable", $serviceName);
	}
}

sub sprintFile(\%$$) {
	my ($configurations_h, $templateFile, $configFile) = @_;
	
	my %configurations = %$configurations_h;
	my $sedCommand = "sed ";

	for my $conf (keys %configurations) {
		$sedCommand = "$sedCommand -e 's|$conf|$configurations{$conf}|'";
	}	
 
	open (TEMPOUT, "> $configFile");
	print TEMPOUT `$sedCommand $templateFile`;
	close TEMPOUT;
}




