"""
Version 3.0 - needs BlueCat Micetro version >= 11.0.0
    Copyright (c) 2024 BlueCat Networks
A python >=3.10 script that reads env variables.
Gets executed by the genericDNSPowerDNSP3.py Micetro Generic DNS Controller Python code.
Please edit the systemd startup mmremote.service script to export the following variables:

Environment=BASE_URL="http://localhost:8081/"   # your PDNS REST API base URL
Environment=SERVER="localhost" # the server instance name
Environment=API_KEY="bluecat123" # the API secret configured in the PDNS config

# The following entries are optional and default to the values listed below,
# but can be overwritten
Environment=PDNS_DEFAULT_TTL="3600" # your DEFAULT TTL value
Environment=LOG_LEVEL="INFO" # INFO or DEBUG is supported as log level
Environment=LOG_FILE_NAME="/tmp/mmGenericDNSPowerDNS.log" # log file path
Environment=PDNS_MODE="NATIVE" # mode that is configured in the pdns.conf

or create a /etc/sysconfig/mmremote
BASE_URL="http://localhost:8081/"   # your PDNS REST API base URL
SERVER="localhost" # the server instance name
API_KEY="bluecat123" # the API secret configured in the PDNS config

# again the optional env variables that can also be overwritten
PDNS_DEFAULT_TTL="3600" # your DEFAULT TTL value
LOG_LEVEL="INFO" # INFO or DEBUG is supported as log level
LOG_FILE_NAME="/tmp/mmGenericDNSPowerDNS.log" # log file path
PDNS_MODE="NATIVE" # mode that is configured in the pdns.conf

and then use the sysconfig/mmremote file in the systemd startup script of mmremote.service:

EnvironmentFile=-/etc/sysconfig/mmremote
"""
import os
from dotenv import load_dotenv

REQUIRED_VARS = ["API_KEY", "SERVER", "BASE_URL"]
OPTIONAL_VARS = ["PDNS_DEFAULT_TTL", "LOG_LEVEL", "LOG_FILE_NAME", "PDNS_MODE"]


class PdnsMock:
    """PDNS Mock Class"""
    def __init__(self, base_url: str, server: str, api_key: str, pdns_default_ttl: str,
                 log_level: str, log_file_name: str, pdns_mode: str):
        self.base_url = base_url
        self.server = server
        self.api_key = api_key
        self.pdns_default_ttl = pdns_default_ttl
        self.log_level = log_level.upper()
        self.log_file_name = log_file_name
        self.pdns_mode = pdns_mode


def validate_vars() -> dict:
    """Validates the needed vars have been loaded from the dotenv file, and returns a dict"""
    response = {"err": None, "values": {}}
    for var in REQUIRED_VARS:
        if os.environ.get(var) is not None:
            response['values'][var.lower()] = str(os.environ.get(var))
        else:
            response['err'] = ValueError(
                f"The Environmental variable for {var} is missing")

    for var in OPTIONAL_VARS:
        if os.environ.get(var) is not None:
            response['values'][var.lower()] = str(os.environ.get(var))

    return response


def load_vars() -> PdnsMock:
    """entry Point"""
    environment = "dev"  # Normally acquired from an envar or sys.argv
    env_file = f".env.{environment}"
    load_dotenv(env_file)

    # validate that all the needed vars are available
    vars: dict = validate_vars()
    if vars['err'] is not None:
        raise vars['err']
    pdns_mode = "NATIVE"
    if "pdns_mode" in vars['values'].keys():
        pdns_mode = vars['values']["pdns_mode"]

    log_file_name = "/tmp/mmGenericDNSPowerDNS.log"
    if "log_file_name" in vars['values'].keys():
        log_file_name = vars['values']["log_file_name"]

    log_level = "INFO"
    if "log_level" in vars['values'].keys():
        log_level = vars['values']["log_level"]

    pdns_default_ttl = "3600"
    if "pdns_default_ttl" in vars['values'].keys():
        pdns_default_ttl = vars['values']["pdns_default_ttl"]

    dns = PdnsMock(
        base_url=vars['values']["base_url"],
        server=vars['values']["server"],
        api_key=vars['values']["api_key"],
        pdns_default_ttl=pdns_default_ttl,
        log_level=log_level,
        log_file_name=log_file_name,
        pdns_mode=pdns_mode
    )
    return dns
